// API-based category service for admin dashboard
import { buildAuthHeaders } from '../utils/apiClient';

const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:3001/api';

export interface Category {
  id: string;
  name: string;
  slug: string;
  description: string;
  parentId: string | null;
  productCount: number;
  imageUrl?: string;
  subcategories?: Category[];
}

// In-memory cache for categories (keyed by catalogId)
let categoriesCache: Record<string, Category[]> = {};
let cacheTimestamps: Record<string, number> = {};
const CACHE_DURATION = 5000; // 5 seconds


// Category change listeners
type CategoryChangeListener = (categories: Category[]) => void;
const listeners: CategoryChangeListener[] = [];

export const categoryService = {
  /**
   * Fetch all categories from API
   */
  async fetchAll(catalogId?: string | null): Promise<Category[]> {
    try {
      const response = await fetch(`${API_URL}/categories`, {
        headers: buildAuthHeaders({ catalogId }),
      });

      if (!response.ok) {
        throw new Error('Failed to fetch categories');
      }

      const data = await response.json();
      const categories = data.data || data;

      // Update cache
      const key = catalogId || 'default';
      categoriesCache[key] = categories;
      cacheTimestamps[key] = Date.now();

      return categories;
    } catch (error) {
      console.error('Error fetching categories:', error);
      return [];
    }
  },


  /**
   * Get all categories (from cache or fetch)
   */
  async getAll(catalogId?: string | null): Promise<Category[]> {
    const key = catalogId || 'default';

    // Return cache if fresh
    if (categoriesCache[key] && Date.now() - (cacheTimestamps[key] || 0) < CACHE_DURATION) {
      return [...categoriesCache[key]];
    }

    // Fetch fresh data
    return await this.fetchAll(catalogId);
  },


  /**
   * Get all categories synchronously (from cache only)
   */
  getAllSync(catalogId?: string | null): Category[] {
    const key = catalogId || 'default';
    return [...(categoriesCache[key] || [])];
  },


  /**
   * Get flattened list of all categories (including subcategories)
   */
  getAllFlattened(catalogId?: string | null): Category[] {
    const flattened: Category[] = [];
    const key = catalogId || 'default';
    const cats = categoriesCache[key] || [];

    const flatten = (categories: Category[]) => {
      categories.forEach(cat => {
        flattened.push(cat);
        if (cat.subcategories) {
          flatten(cat.subcategories);
        }
      });
    };

    flatten(cats);
    return flattened;
  },


  /**
   * Get category by ID
   */
  getById(id: string, catalogId?: string | null): Category | undefined {
    const flattened = this.getAllFlattened(catalogId);
    return flattened.find(cat => cat.id === id);
  },


  /**
   * Add new category via API
   */
  async add(category: Omit<Category, 'id' | 'productCount'>, catalogId?: string | null): Promise<Category> {
    try {
      const response = await fetch(`${API_URL}/categories`, {
        method: 'POST',
        headers: buildAuthHeaders({ catalogId }),
        body: JSON.stringify(category),
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error?.message || 'Failed to create category');
      }

      const data = await response.json();
      const newCategory = data.data;

      // Refresh cache
      await this.fetchAll(catalogId);
      this.notifyListeners(catalogId);

      return newCategory;
    } catch (error) {
      console.error('Error creating category:', error);
      throw error;
    }
  },


  /**
   * Update category via API
   */
  async update(id: string, updates: Partial<Category>, catalogId?: string | null): Promise<Category> {
    try {
      const response = await fetch(`${API_URL}/categories/${id}`, {
        method: 'PUT',
        headers: buildAuthHeaders({ catalogId }),
        body: JSON.stringify(updates),
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error?.message || 'Failed to update category');
      }

      const data = await response.json();
      const updatedCategory = data.data;

      // Refresh cache
      await this.fetchAll(catalogId);
      this.notifyListeners(catalogId);

      return updatedCategory;
    } catch (error) {
      console.error('Error updating category:', error);
      throw error;
    }
  },


  /**
   * Delete category via API
   */
  async delete(id: string, catalogId?: string | null): Promise<void> {
    try {
      const response = await fetch(`${API_URL}/categories/${id}`, {
        method: 'DELETE',
        headers: buildAuthHeaders({ includeJson: false, catalogId }),
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error?.message || 'Failed to delete category');
      }

      // Refresh cache
      await this.fetchAll(catalogId);
      this.notifyListeners(catalogId);
    } catch (error) {
      console.error('Error deleting category:', error);
      throw error;
    }
  },


  /**
   * Set all categories (for initial load or reset)
   */
  setAll(categories: Category[], catalogId?: string | null): void {
    const key = catalogId || 'default';
    categoriesCache[key] = categories;
    cacheTimestamps[key] = Date.now();
    this.notifyListeners(catalogId);
  },

  /**
   * Clear cache (force refresh on next fetch)
   */
  clearCache(): void {
    categoriesCache = {};
    cacheTimestamps = {};
  },

  /**
   * Subscribe to category changes
   */
  subscribe(listener: CategoryChangeListener): () => void {
    listeners.push(listener);
    // Return unsubscribe function
    return () => {
      const index = listeners.indexOf(listener);
      if (index > -1) {
        listeners.splice(index, 1);
      }
    };
  },

  /**
   * Notify all listeners of changes
   */
  notifyListeners(catalogId?: string | null): void {
    const categories = this.getAllSync(catalogId);
    listeners.forEach(listener => listener(categories));
  },

};
