import React, { useEffect, useState } from 'react';
import { Blocks, ToggleLeft, ToggleRight, Loader } from 'lucide-react';
import { pluginService, Plugin } from './plugin.service';

const PluginsPage: React.FC = () => {
    const [plugins, setPlugins] = useState<Plugin[]>([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState<string | null>(null);

    useEffect(() => {
        loadPlugins();
    }, []);

    const loadPlugins = async () => {
        try {
            const data = await pluginService.getPlugins();
            setPlugins(data);
        } catch (err) {
            setError('Failed to load plugins');
        } finally {
            setLoading(false);
        }
    };

    const handleToggle = async (plugin: Plugin) => {
        try {
            await pluginService.togglePlugin(plugin.name);
            // Reload to reflect changes
            await loadPlugins();
            // Optionally show toast "Restart recommended"
        } catch (err) {
            setError('Failed to toggle plugin');
        }
    };

    const handleUpload = async (file: File) => {
        if (!file.name.endsWith('.zip')) {
            setError('Please upload a .zip file');
            return;
        }

        try {
            setLoading(true); // Using global loading or local loading state
            await pluginService.uploadPlugin(file);
            await loadPlugins();
            alert('Plugin uploaded successfully! Please restart the server to verify the new plugin.');
        } catch (err: any) {
            setError(err.message || 'Failed to upload plugin');
        } finally {
            setLoading(false);
        }
    };

    const onDrop = (e: React.DragEvent) => {
        e.preventDefault();
        const file = e.dataTransfer.files[0];
        if (file) handleUpload(file);
    };

    const onFileSelect = (e: React.ChangeEvent<HTMLInputElement>) => {
        const file = e.target.files?.[0];
        if (file) handleUpload(file);
    };

    if (loading) return (
        <div className="flex justify-center items-center h-full">
            <Loader className="w-8 h-8 animate-spin text-blue-600" />
        </div>
    );

    return (
        <div className="p-6">
            <div className="mb-6">
                <h1 className="text-2xl font-bold flex items-center gap-2">
                    <Blocks className="w-8 h-8" />
                    Extension Plugins
                </h1>
                <p className="text-gray-500">Manage system extensions and features.</p>
            </div>

            {error && (
                <div className="bg-red-50 text-red-600 p-4 rounded-lg mb-6">
                    {error}
                </div>
            )}

            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                {plugins.map((plugin) => (
                    <div key={plugin.name} className="bg-white rounded-xl shadow-sm border border-gray-100 p-6">
                        <div className="flex justify-between items-start mb-4">
                            <div className="p-3 bg-blue-50 rounded-lg">
                                <Blocks className="w-6 h-6 text-blue-600" />
                            </div>
                            <div className="flex flex-col items-end">
                                <span className={`px-2 py-1 text-xs rounded-full font-medium ${plugin.status === 'ACTIVE'
                                    ? 'bg-green-100 text-green-700'
                                    : 'bg-gray-100 text-gray-600'
                                    }`}>
                                    {plugin.status}
                                </span>
                            </div>
                        </div>

                        <h3 className="text-lg font-semibold text-gray-900 mb-1">{plugin.name}</h3>
                        <div className="text-sm text-gray-500 mb-4">Version {plugin.version}</div>
                        <p className="text-gray-600 text-sm mb-6">{plugin.description || 'System Plugin'}</p>

                        <div className="flex items-center justify-between pt-4 border-t border-gray-50">
                            <span className="text-xs text-gray-400">Core Plugin</span>
                            <button
                                onClick={() => handleToggle(plugin)}
                                className={`transition-colors ${plugin.status === 'ACTIVE' ? 'text-green-500 hover:text-green-600' : 'text-gray-300 hover:text-gray-400'
                                    }`}
                            >
                                {plugin.status === 'ACTIVE' ? (
                                    <ToggleRight className="w-8 h-8" />
                                ) : (
                                    <ToggleLeft className="w-8 h-8" />
                                )}
                            </button>
                        </div>
                    </div>
                ))}

                {/* Upload Section */}
                <label
                    onDragOver={(e) => e.preventDefault()}
                    onDrop={onDrop}
                    className="border-2 border-dashed border-gray-200 rounded-xl p-6 flex flex-col items-center justify-center text-center hover:border-blue-300 hover:bg-blue-50 transition-all cursor-pointer bg-white group"
                >
                    <input type="file" className="hidden" accept=".zip" onChange={onFileSelect} />
                    <div className="p-3 bg-gray-50 rounded-full mb-3 shadow-sm group-hover:bg-white transition-colors">
                        <Blocks className="w-6 h-6 text-gray-400 group-hover:text-blue-500 transition-colors" />
                    </div>
                    <h3 className="font-medium text-gray-900 group-hover:text-blue-600">Upload Plugin</h3>
                    <p className="text-sm text-gray-500 mt-1">Drag .zip file here or click to browse</p>
                </label>
            </div>
        </div>
    );
};

export default PluginsPage;
