import { RefreshCcw } from 'lucide-react';
import { useMemo } from 'react';
import { useCatalog } from '../contexts/CatalogContext';

const CatalogSelector = () => {
  const {
    catalogs,
    selectedCatalogId,
    setSelectedCatalogId,
    refreshCatalogs,
    isLoading,
    error,
  } = useCatalog();

  const hasCatalogs = catalogs.length > 0;
  const selectValue = hasCatalogs ? selectedCatalogId ?? '' : '';

  const statusLabel = useMemo(() => {
    if (isLoading && !hasCatalogs) {
      return 'Loading catalogs…';
    }

    if (!hasCatalogs) {
      return error ?? 'No catalogs found';
    }

    const current = catalogs.find(catalog => catalog.id === selectedCatalogId);
    return current ? `Active catalog: ${current.name}` : 'Select a catalog';
  }, [catalogs, error, hasCatalogs, isLoading, selectedCatalogId]);

  const handleSelectChange = (event: React.ChangeEvent<HTMLSelectElement>) => {
    setSelectedCatalogId(event.target.value);
  };

  return (
    <div className="flex flex-col lg:flex-row lg:items-center lg:gap-3 w-full lg:w-auto">
      <div className="text-xs text-gray-500 uppercase tracking-wide">Catalog</div>
      <div className="flex items-center gap-2">
        <select
          value={selectValue}
          onChange={handleSelectChange}
          disabled={!hasCatalogs || isLoading}
          className="min-w-[200px] px-3 py-2 border border-gray-300 rounded-lg bg-white text-sm focus:ring-2 focus:ring-blue-500 focus:border-transparent disabled:bg-gray-100 disabled:text-gray-500"
        >
          {hasCatalogs ? (
            catalogs.map(catalog => (
              <option key={catalog.id} value={catalog.id}>
                {catalog.name}
                {catalog.isDefault ? ' • Default' : ''}
                {!catalog.isActive ? ' • Inactive' : ''}
              </option>
            ))
          ) : (
            <option value="">{isLoading ? 'Loading…' : 'No catalogs available'}</option>
          )}
        </select>
        <button
          type="button"
          onClick={refreshCatalogs}
          className="p-2 rounded-lg border border-gray-200 text-gray-500 hover:text-gray-700 hover:border-gray-300"
          title="Refresh catalogs"
        >
          <RefreshCcw className="w-4 h-4" />
        </button>
      </div>
      <p className="text-xs text-gray-500 mt-1 lg:mt-0">{statusLabel}</p>
    </div>
  );
};

export default CatalogSelector;
