import { useState, useEffect } from 'react';
import { CreditCard, Truck, FileText, AlertCircle, ArrowLeft, Building2, User } from 'lucide-react';
import { useLanguage } from '../contexts/LanguageContext';
import { formatPrice } from '../utils/currency';
import { useAuth } from '../context/AuthContext';
import { quotationService } from '../services/quotationService';
import { CartItem } from '../context/CartContext';

interface CheckoutScreenProps {
  cartItems: CartItem[];
  onBack: () => void;
  onOrderComplete: (orderData: any) => void;
  onUpdateQuantity: (productId: string, quantity: number) => void;
  onRemoveItem: (productId: string) => void;
}

export default function CheckoutScreen({
  cartItems,
  onBack,
  onOrderComplete,
  onUpdateQuantity,
  onRemoveItem
}: CheckoutScreenProps) {
  const { user } = useAuth();
  const { language } = useLanguage();
  const [isProcessing, setIsProcessing] = useState(false);
  const [isLoadingUser, setIsLoadingUser] = useState(true);
  const [notes, setNotes] = useState('');

  // Helper function to format payment method display
  const formatPaymentMethod = (method: string | undefined) => {
    if (!method) return language === 'fr' ? 'Non attribué' : 'Not assigned';

    const paymentMethodLabels: Record<string, { en: string; fr: string }> = {
      // PaymentTerm enum values from database
      'CASH_ON_DELIVERY': { en: 'Cash on Delivery', fr: 'Paiement à la Livraison' },
      'CHEQUE_ON_DELIVERY': { en: 'Cheque on Delivery', fr: 'Chèque à la Livraison' },
      'NET_30': { en: 'Payment on Due Date (NET 30)', fr: 'Paiement à Échéance (NET 30)' },
      'NET_60': { en: 'Payment on Due Date (NET 60)', fr: 'Paiement à Échéance (NET 60)' },
      'NET_90': { en: 'Payment on Due Date (NET 90)', fr: 'Paiement à Échéance (NET 90)' },
      'NET_120': { en: 'Payment on Due Date (NET 120)', fr: 'Paiement à Échéance (NET 120)' },
      // Legacy values for backward compatibility
      'COD': { en: 'Cash on Delivery', fr: 'Paiement à la Livraison' },
      'CHEQUE': { en: 'Cheque on Delivery', fr: 'Chèque à la Livraison' },
      'NET_TERMS': { en: 'Payment on Due Date', fr: 'Paiement à Échéance' },
      'IMMEDIATE': { en: 'Cash on Delivery', fr: 'Paiement à la Livraison' },
    };

    return paymentMethodLabels[method]?.[language] || method;
  };

  // Helper function to format account status
  const formatAccountStatus = (status: string | undefined) => {
    if (!status) return language === 'fr' ? 'Non défini' : 'Not defined';

    const statusLabels: Record<string, { en: string; fr: string }> = {
      'COMMERCIAL_IN_PROCESS': { en: 'Commercial In Process', fr: 'COMMERCIAL_IN_PROCESS' },
      'FINANCIAL_IN_PROCESS': { en: 'Financial In Process', fr: 'FINANCIAL_IN_PROCESS' },
      'ACTIVE': { en: 'Active', fr: 'ACTIVE' },
      'FINANCIAL_NON_CURRENT': { en: 'Financial Non-Current', fr: 'FINANCIAL_NON_CURRENT' },
      'SUSPENDED': { en: 'Suspended', fr: 'SUSPENDED' },
    };

    return statusLabels[status]?.[language] || status;
  };

  // Helper function to format payment terms (deprecated, now part of paymentTerm enum)
  const formatPaymentTerms = (terms: string | undefined) => {
    if (!terms) return '';

    const termsLabels: Record<string, { en: string; fr: string }> = {
      'NET_30': { en: 'NET 30', fr: 'NET 30' },
      'NET_60': { en: 'NET 60', fr: 'NET 60' },
      'NET_90': { en: 'NET 90', fr: 'NET 90' },
      'NET_120': { en: 'NET 120', fr: 'NET 120' },
    };

    return termsLabels[terms]?.[language] || terms;
  };

  // Helper function to get complete payment condition label
  const getPaymentConditionLabel = () => {
    // Use paymentTerm from user object (this is the PaymentTerm enum from database)
    const paymentTerm = user?.paymentTerm;

    return formatPaymentMethod(paymentTerm);
  };

  // Load user data on mount
  useEffect(() => {
    if (user) {
      setIsLoadingUser(false);
    }
  }, [user]);

  // Calculate TVA by rate with discounts
  const calculateTvaBreakdown = () => {
    let tva19 = 0;
    let tva7 = 0;
    let tva0 = 0;

    let base19 = 0;
    let base7 = 0;
    let base0 = 0;

    let totalBrutHT = 0;
    let totalRemise = 0;

    cartItems.forEach(item => {
      const product = item.product;
      const now = new Date();

      // Check if discount is active
      const isDiscountActive = product.hasDiscount &&
        product.discountValue &&
        product.discountValue > 0 &&
        (!product.discountStartDate || new Date(product.discountStartDate) <= now) &&
        (!product.discountEndDate || new Date(product.discountEndDate) >= now);

      const basePrice = product.basePrice || product.price || 0;
      const lineBrutHT = basePrice * item.quantity;
      totalBrutHT += lineBrutHT;

      // Calculate discount
      let lineDiscount = 0;
      if (isDiscountActive && product.discountValue) {
        if (product.discountType === 'percentage') {
          lineDiscount = (lineBrutHT * product.discountValue) / 100;
        } else if (product.discountType === 'fixed') {
          lineDiscount = Math.min(product.discountValue * item.quantity, lineBrutHT);
        }
      }
      totalRemise += lineDiscount;

      // Calculate net HT after discount
      const lineNetHT = lineBrutHT - lineDiscount;

      // Calculate TVA on net amount
      const tvaRate = product.tvaExempt ? 0 : (product.tvaRate || 19);
      const lineTVA = (lineNetHT * tvaRate) / 100;

      if (tvaRate === 19) {
        tva19 += lineTVA;
        base19 += lineNetHT;
      } else if (tvaRate === 7) {
        tva7 += lineTVA;
        base7 += lineNetHT;
      } else {
        tva0 += lineTVA; // Always 0, but good for consistency
        base0 += lineNetHT;
      }
    });

    const totalNetHT = totalBrutHT - totalRemise;
    const totalTVA = tva19 + tva7 + tva0;
    const timbreFiscal = 1.000;
    const netAPayer = totalNetHT + totalTVA + timbreFiscal;

    return {
      totalBrutHT,
      totalRemise,
      totalNetHT,
      tva19,
      tva7,
      tva0,
      base19,
      base7,
      base0,
      totalTVA,
      timbreFiscal,
      netAPayer
    };
  };

  const tvaBreakdown = calculateTvaBreakdown();
  const deliveryFee = 0; // No delivery fee for B2B
  const total = tvaBreakdown.netAPayer;


  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!user) {
      alert('Please login to place an order');
      return;
    }

    // Validate cart is not empty
    if (!cartItems || cartItems.length === 0) {
      alert('Your cart is empty. Please add items before placing an order.');
      onBack();
      return;
    }

    setIsProcessing(true);

    try {
      // All customers submit quotations
      const quotationData = {
        items: cartItems.map(item => ({
          productId: item.product.id,
          quantity: item.quantity,
        })),
        notes: ''
      };

      const result = await quotationService.create(quotationData);

      if (result) {
        // Quotation created successfully
        onOrderComplete({
          id: result.id,
          orderNumber: result.quotationNumber,
          isQuotation: true,
          customer: {
            name: `${user.firstName || ''} ${user.lastName || ''}`.trim() || user.email,
            email: user.email,
            phone: user.phone || 'N/A',
            company: user.companyName
          },
          items: cartItems,
          totals: {
            subtotal: tvaBreakdown.totalNetHT,
            tax: tvaBreakdown.totalTVA,
            delivery: deliveryFee,
            total
          },
          paymentMethod: user.paymentTerm || user.paymentMethod || 'NET_TERMS',
          status: result.status || 'PENDING_APPROVAL'
        });
      } else {
        throw new Error('Failed to create quotation');
      }
    } catch (error: any) {
      console.error('Quotation creation failed:', error);
      // Show specific error message from backend if available (e.g. "Insufficient stock")
      alert(error.message || 'Échec de la soumission du devis. Veuillez réessayer.');
    } finally {
      setIsProcessing(false);
    }
  };

  if (cartItems.length === 0) {
    return (
      <div className="flex flex-col items-center justify-center px-4 py-20">
        <div className="w-24 h-24 bg-gray-100 rounded-full flex items-center justify-center mb-4">
          <svg
            className="w-12 h-12 text-gray-400"
            fill="none"
            stroke="currentColor"
            viewBox="0 0 24 24"
          >
            <path
              strokeLinecap="round"
              strokeLinejoin="round"
              strokeWidth={2}
              d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"
            />
          </svg>
        </div>
        <h2 className="text-xl font-semibold text-gray-800 mb-2">Aucun article à commander</h2>
        <p className="text-gray-500 text-center mb-4">Ajoutez d'abord des produits à votre panier!</p>
        <button
          onClick={onBack}
          className="bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700"
        >
          Retour aux achats
        </button>
      </div>
    );
  }

  return (
    <div className="pb-40">
      <div className="px-4 py-4">
        <div className="flex items-center gap-4 mb-6">
          <button
            onClick={onBack}
            className="p-2 hover:bg-gray-100 rounded-lg transition-colors"
          >
            <ArrowLeft className="w-5 h-5" />
          </button>
          <h1 className="text-xl font-semibold text-gray-900">Finaliser la commande</h1>
        </div>

        <form onSubmit={handleSubmit} className="space-y-6">
          {/* Customer Information - Display Only */}
          <div className="bg-white rounded-lg shadow-sm p-4">
            <div className="flex items-center gap-2 mb-4">
              <User className="w-5 h-5 text-gray-600" />
              <h2 className="text-lg font-semibold">Informations client</h2>
            </div>

            {isLoadingUser ? (
              <div className="text-center py-8">
                <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto"></div>
                <p className="text-gray-600 mt-2">Chargement de vos informations...</p>
              </div>
            ) : user ? (
              <div className="space-y-3">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-600 mb-1">
                      Nom complet
                    </label>
                    <p className="text-gray-900 font-medium">
                      {user.firstName} {user.lastName}
                    </p>
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-600 mb-1">
                      Email
                    </label>
                    <p className="text-gray-900">{user.email}</p>
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-600 mb-1">
                      Téléphone
                    </label>
                    <p className="text-gray-900">{user.phone || 'Non fourni'}</p>
                  </div>

                  {user.companyName && (
                    <div>
                      <label className="block text-sm font-medium text-gray-600 mb-1">
                        Entreprise
                      </label>
                      <div className="flex items-center gap-2">
                        <Building2 className="w-4 h-4 text-gray-400" />
                        <p className="text-gray-900">{user.companyName}</p>
                      </div>
                    </div>
                  )}

                  {user.customerType && (
                    <div>
                      <label className="block text-sm font-medium text-gray-600 mb-1">
                        Type de client
                      </label>
                      <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                        {user.customerType}
                      </span>
                    </div>
                  )}
                </div>

                <div className="mt-4 p-3 bg-blue-50 border border-blue-200 rounded-lg">
                  <p className="text-sm text-blue-800">
                    ✓ Vos informations de compte seront utilisées pour cette commande
                  </p>
                </div>
              </div>
            ) : (
              <div className="text-center py-8">
                <p className="text-red-600">Veuillez vous connecter pour continuer</p>
              </div>
            )}
          </div>

          {/* Shipping Address Note */}
          <div className="bg-white rounded-lg shadow-sm p-4">
            <div className="flex items-center gap-2 mb-4">
              <Truck className="w-5 h-5 text-gray-600" />
              <h2 className="text-lg font-semibold">Adresse de livraison</h2>
            </div>
            <div className="p-3 bg-gray-50 border border-gray-200 rounded-lg">
              <p className="text-sm text-gray-700">
                Votre adresse de livraison par défaut sera utilisée pour cette commande.
                <br />
                <span className="text-gray-600">Vous pouvez gérer vos adresses dans les paramètres de votre compte.</span>
              </p>
            </div>
          </div>

          {/* Payment Condition - Read Only */}
          <div className="bg-white rounded-lg shadow-sm p-4">
            <div className="flex items-center gap-2 mb-4">
              <FileText className="w-5 h-5 text-gray-600" />
              <h2 className="text-lg font-semibold">
                {language === 'fr' ? 'Condition de paiement' : 'Payment Condition'}
              </h2>
            </div>

            <div className="p-4 bg-blue-50 border border-blue-200 rounded-lg">
              <div className="flex items-center gap-3">
                <CreditCard className="w-5 h-5 text-blue-600" />
                <div className="flex-1">
                  <p className="text-sm text-gray-600 mb-1">
                    {language === 'fr' ? 'Condition de paiement :' : 'Payment Condition:'}
                  </p>
                  <p className="text-lg font-semibold text-gray-900">
                    {getPaymentConditionLabel()}
                  </p>
                </div>
              </div>
            </div>

            {/* Account Status Warning */}
            {user?.accountStatus && user.accountStatus !== 'ACTIVE' && (
              <div className="mt-3 p-4 bg-yellow-50 border border-yellow-200 rounded-lg flex items-start gap-3">
                <AlertCircle className="w-5 h-5 text-yellow-600 mt-0.5" />
                <div>
                  <h4 className="font-medium text-yellow-900 mb-1">
                    {language === 'fr' ? 'Statut du compte' : 'Account Status'}: {formatAccountStatus(user.accountStatus)}
                  </h4>
                  <p className="text-sm text-yellow-800">
                    {language === 'fr'
                      ? 'Votre devis nécessitera une approbation administrative.'
                      : 'Your quotation will require administrative approval.'}
                  </p>
                </div>
              </div>
            )}

            {/* Credit Limit Warning */}
            {user?.financialLimit && (user.currentOutstanding !== undefined || user.outstandingBalance !== undefined) &&
              ((user.currentOutstanding || user.outstandingBalance || 0) + total) > user.financialLimit && (
                <div className="mt-3 p-4 bg-red-50 border border-red-200 rounded-lg flex items-start gap-3">
                  <AlertCircle className="w-5 h-5 text-red-600 mt-0.5" />
                  <div>
                    <h4 className="font-medium text-red-900 mb-1">
                      {language === 'fr' ? 'Limite de crédit dépassée' : 'Credit Limit Exceeded'}
                    </h4>
                    <p className="text-sm text-red-800">
                      {language === 'fr'
                        ? 'Cette commande dépassera votre limite de crédit. Une approbation administrative sera requise.'
                        : 'This order will exceed your credit limit. Administrative approval will be required.'}
                    </p>
                    <p className="text-xs text-red-700 mt-1">
                      {language === 'fr' ? 'Nouveau solde' : 'New Outstanding'}: {formatPrice((user.currentOutstanding || user.outstandingBalance || 0) + total)} /
                      {language === 'fr' ? ' Limite' : ' Limit'}: {formatPrice(user.financialLimit)}
                    </p>
                  </div>
                </div>
              )}
          </div>
        </form>

        {/* Professional Quotation Document Preview */}
        <div className="bg-white p-6 max-w-[210mm] mx-auto shadow-lg" style={{ fontFamily: 'Arial, sans-serif' }}>
          {/* HEADER SECTION - 3 Columns */}
          <div className="grid grid-cols-3 gap-4 mb-6 pb-4 border-b-2 border-gray-300">
            {/* LEFT: Company Info */}
            <div className="space-y-1">
              <h1 className="text-base font-bold text-gray-900">SMD Tunisie</h1>
              <p className="text-xs text-gray-700">Vente Produits Sanitaires & Quincaillerie</p>
              <p className="text-xs text-gray-600">123 Avenue Habib Bourguiba</p>
              <p className="text-xs text-gray-600">Tunis, 1000</p>
              <p className="text-xs text-gray-600">+216 71 123 456</p>
              <p className="text-xs text-gray-600">contact@smd-tunisie.com</p>
              <p className="text-xs text-gray-500 mt-2">Date: {new Date().toLocaleDateString('fr-TN')}</p>
            </div>

            {/* CENTER: Logo */}
            <div className="flex items-center justify-center">
              <div className="w-24 h-24 bg-gray-100 rounded-lg flex items-center justify-center">
                <span className="text-lg font-bold text-gray-400">LOGO</span>
              </div>
            </div>

            {/* RIGHT: Legal & Fiscal Info */}
            <div className="space-y-1 text-right">
              <p className="text-xs text-gray-600">
                <span className="font-semibold">Ident. Fiscal:</span> 1234567/A/M/000
              </p>
              <p className="text-xs text-gray-600">
                <span className="font-semibold">RC:</span> B123456789
              </p>
              <p className="text-xs text-gray-600 mt-2">
                <span className="font-semibold">Date:</span> {new Date().toLocaleDateString('fr-TN')}
              </p>
              <div className="mt-4 p-2 bg-blue-50 border border-blue-200 rounded">
                <p className="text-sm font-bold text-blue-900">DEVIS (Brouillon)</p>
              </div>
            </div>
          </div>

          {/* CLIENT INFORMATION BLOCK */}
          <div className="border-2 border-gray-300 rounded p-4 mb-6">
            <div className="grid grid-cols-2 gap-x-8 gap-y-2">
              <div>
                <span className="font-semibold text-sm">Client :</span>
                <span className="text-sm ml-2">
                  {user?.firstName} {user?.lastName}
                  {user?.companyName && ` - ${user.companyName}`}
                </span>
              </div>
              <div>
                <span className="font-semibold text-sm">Ident. Fisc. :</span>
                <span className="text-sm ml-2">{(user as any)?.taxId || 'N/A'}</span>
              </div>
              <div className="col-span-2">
                <span className="font-semibold text-sm">Adresse :</span>
                <span className="text-sm ml-2">{(user as any)?.address || (user as any)?.shippingAddress || 'À définir'}</span>
              </div>
            </div>
          </div>

          {/* PRODUCTS TABLE */}
          <div className="mb-6">
            <table className="w-full border-collapse border border-gray-300 text-xs">
              <thead>
                <tr className="bg-gray-100">
                  <th className="border border-gray-300 p-2 text-left font-bold">Code</th>
                  <th className="border border-gray-300 p-2 text-left font-bold">Article / Désignation</th>
                  <th className="border border-gray-300 p-2 text-center font-bold">Unité</th>
                  <th className="border border-gray-300 p-2 text-right font-bold">Qté</th>
                  <th className="border border-gray-300 p-2 text-right font-bold">Prix Unit HT</th>
                  <th className="border border-gray-300 p-2 text-right font-bold">Remise</th>
                  <th className="border border-gray-300 p-2 text-right font-bold">Total HT</th>
                  <th className="border border-gray-300 p-2 text-right font-bold">TVA</th>
                  <th className="border border-gray-300 p-2 text-right font-bold">Total TTC</th>
                </tr>
              </thead>
              <tbody>
                {cartItems.map((item, index) => {
                  const product = item.product;
                  const now = new Date();
                  const isDiscountActive = product.hasDiscount &&
                    product.discountValue &&
                    product.discountValue > 0 &&
                    (!product.discountStartDate || new Date(product.discountStartDate) <= now) &&
                    (!product.discountEndDate || new Date(product.discountEndDate) >= now);

                  const basePrice = product.basePrice || product.price || 0;
                  let discountAmount = 0;
                  if (isDiscountActive && product.discountValue) {
                    if (product.discountType === 'percentage') {
                      discountAmount = (basePrice * item.quantity * product.discountValue) / 100;
                    } else if (product.discountType === 'fixed') {
                      discountAmount = Math.min(product.discountValue * item.quantity, basePrice * item.quantity);
                    }
                  }

                  const totalHT = (basePrice * item.quantity) - discountAmount;
                  const tvaRate = product.tvaExempt ? 0 : (product.tvaRate || 19);
                  const tvaAmount = (totalHT * tvaRate) / 100;
                  const totalTTC = totalHT + tvaAmount;

                  return (
                    <tr key={index} className={index % 2 === 0 ? 'bg-white' : 'bg-gray-50'}>
                      <td className="border border-gray-300 p-2">{product.sku || 'N/A'}</td>
                      <td className="border border-gray-300 p-2">{product.name}</td>
                      <td className="border border-gray-300 p-2 text-center">pcs</td>
                      <td className="border border-gray-300 p-2 text-right">{item.quantity}</td>
                      <td className="border border-gray-300 p-2 text-right">{basePrice.toFixed(3)}</td>
                      <td className="border border-gray-300 p-2 text-right">
                        {discountAmount > 0 ? discountAmount.toFixed(3) : '0.000'}
                      </td>
                      <td className="border border-gray-300 p-2 text-right font-semibold">
                        {totalHT.toFixed(3)}
                      </td>
                      <td className="border border-gray-300 p-2 text-right">
                        {product.tvaExempt ? 'Exonéré' : `${tvaRate.toFixed(1)} %`}
                      </td>
                      <td className="border border-gray-300 p-2 text-right font-semibold">
                        {totalTTC.toFixed(3)}
                      </td>
                    </tr>
                  );
                })}
              </tbody>
            </table>
          </div>

          {/* SUMMARY SECTION */}
          <div className="flex justify-end mb-6">
            <div className="w-80 space-y-2 text-sm">
              <div className="flex justify-between py-1 border-b border-gray-200">
                <span className="text-gray-700">Total Brut HT</span>
                <span className="font-semibold">{tvaBreakdown.totalBrutHT.toFixed(3)} TND</span>
              </div>
              <div className="flex justify-between py-1 border-b border-gray-200">
                <span className="text-gray-700">Total Remise</span>
                <span className="font-semibold text-red-600">
                  -{tvaBreakdown.totalRemise.toFixed(3)} TND
                </span>
              </div>
              <div className="flex justify-between py-1 border-b border-gray-200">
                <span className="text-gray-700">Total Net HT</span>
                <span className="font-semibold">{tvaBreakdown.totalNetHT.toFixed(3)} TND</span>
              </div>
              <div className="flex justify-between py-1 border-b border-gray-200">
                <span className="text-gray-700">Total TVA</span>
                <span className="font-semibold">{tvaBreakdown.totalTVA.toFixed(3)} TND</span>
              </div>
              <div className="flex justify-between py-1 border-b border-gray-200">
                <span className="text-gray-700">Timbre Fiscal</span>
                <span className="font-semibold">{tvaBreakdown.timbreFiscal.toFixed(3)} TND</span>
              </div>
              <div className="flex justify-between py-2 border-t-2 border-gray-800 bg-gray-100 px-2 rounded">
                <span className="text-gray-900 font-bold text-base">NET À PAYER</span>
                <span className="font-bold text-base">{tvaBreakdown.netAPayer.toFixed(3)} TND</span>
              </div>
            </div>
          </div>

          {/* TVA DETAIL TABLE */}
          <div className="mb-6">
            <h3 className="text-sm font-bold mb-2">Détail TVA</h3>
            <table className="w-full border-collapse border border-gray-300 text-xs">
              <thead>
                <tr className="bg-gray-100">
                  <th className="border border-gray-300 p-2 text-left font-bold">Libellé</th>
                  <th className="border border-gray-300 p-2 text-right font-bold">Base</th>
                  <th className="border border-gray-300 p-2 text-right font-bold">Taux</th>
                  <th className="border border-gray-300 p-2 text-right font-bold">Montant</th>
                </tr>
              </thead>
              <tbody>
                {tvaBreakdown.tva19 > 0 && (
                  <tr>
                    <td className="border border-gray-300 p-2">TVA 19%</td>
                    <td className="border border-gray-300 p-2 text-right">
                      {tvaBreakdown.base19.toFixed(3)}
                    </td>
                    <td className="border border-gray-300 p-2 text-right">19.0 %</td>
                    <td className="border border-gray-300 p-2 text-right font-semibold">
                      {tvaBreakdown.tva19.toFixed(3)}
                    </td>
                  </tr>
                )}
                {tvaBreakdown.tva7 > 0 && (
                  <tr>
                    <td className="border border-gray-300 p-2">TVA 7%</td>
                    <td className="border border-gray-300 p-2 text-right">
                      {tvaBreakdown.base7.toFixed(3)}
                    </td>
                    <td className="border border-gray-300 p-2 text-right">7.0 %</td>
                    <td className="border border-gray-300 p-2 text-right font-semibold">
                      {tvaBreakdown.tva7.toFixed(3)}
                    </td>
                  </tr>
                )}
                {tvaBreakdown.base0 > 0 && (
                  <tr>
                    <td className="border border-gray-300 p-2">Exonéré</td>
                    <td className="border border-gray-300 p-2 text-right">
                      {tvaBreakdown.base0.toFixed(3)}
                    </td>
                    <td className="border border-gray-300 p-2 text-right">0.0 %</td>
                    <td className="border border-gray-300 p-2 text-right font-semibold">0.000</td>
                  </tr>
                )}
                <tr className="bg-gray-100 font-bold">
                  <td className="border border-gray-300 p-2">TOTAL</td>
                  <td className="border border-gray-300 p-2 text-right">
                    {tvaBreakdown.totalNetHT.toFixed(3)}
                  </td>
                  <td className="border border-gray-300 p-2 text-right">-</td>
                  <td className="border border-gray-300 p-2 text-right">
                    {tvaBreakdown.totalTVA.toFixed(3)}
                  </td>
                </tr>
              </tbody>
            </table>
          </div>

          {/* NOTES SECTION */}
          <div className="mb-6">
            <label className="block text-sm font-semibold text-gray-700 mb-2">
              Notes (optionnel)
            </label>
            <textarea
              value={notes}
              onChange={(e) => setNotes(e.target.value)}
              placeholder="Ajoutez des notes ou instructions spéciales..."
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent outline-none transition"
              rows={3}
            />
          </div>

          {/* SUBMIT BUTTON */}
          <div className="flex gap-4">
            <button
              onClick={onBack}
              className="flex-1 px-6 py-3 border-2 border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors font-medium"
            >
              Retour au panier
            </button>
            <button
              onClick={handleSubmit}
              disabled={isProcessing}
              className="flex-1 px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:bg-gray-300 disabled:cursor-not-allowed transition-colors font-bold text-lg"
            >
              {isProcessing ? 'Envoi en cours...' : 'Soumettre le devis'}
            </button>
          </div>
        </div>
      </div>
    </div>
  );
}
