interface QuotationDocumentProps {
    quotation: any; // Full quotation object with items and user
}

export default function QuotationDocument({ quotation }: QuotationDocumentProps) {
    const companyInfo = {
        name: 'SMD Tunisie',
        activity: 'Vente Produits Sanitaires & Quincaillerie',
        address: '123 Avenue Habib Bourguiba',
        city: 'Tunis',
        postalCode: '1000',
        phones: ['+216 71 123 456', '+216 98 765 432'],
        email: 'contact@smd-tunisie.com',
        fiscalId: '1234567/A/M/000',
        rc: 'B123456789',
    };

    const formatDate = (date: Date | string) => {
        return new Date(date).toLocaleDateString('fr-TN', {
            year: 'numeric',
            month: '2-digit',
            day: '2-digit',
        });
    };

    const formatPrice = (price: number) => {
        return price.toFixed(3);
    };

    return (
        <div className="bg-white p-8 max-w-[210mm] mx-auto" style={{ fontFamily: 'Arial, sans-serif' }}>
            {/* HEADER SECTION - 3 Columns */}
            <div className="grid grid-cols-3 gap-4 mb-8 pb-4 border-b-2 border-gray-300">
                {/* LEFT: Company Info */}
                <div className="space-y-1">
                    <h1 className="text-lg font-bold text-gray-900">{companyInfo.name}</h1>
                    <p className="text-sm text-gray-700">{companyInfo.activity}</p>
                    <p className="text-xs text-gray-600">{companyInfo.address}</p>
                    <p className="text-xs text-gray-600">{companyInfo.city}, {companyInfo.postalCode}</p>
                    {companyInfo.phones.map((phone, i) => (
                        <p key={i} className="text-xs text-gray-600">{phone}</p>
                    ))}
                    <p className="text-xs text-gray-600">{companyInfo.email}</p>
                    <p className="text-xs text-gray-500 mt-2">Date: {formatDate(quotation.createdAt)}</p>
                    <p className="text-xs text-gray-500">Page: 1/1</p>
                </div>

                {/* CENTER: Logo */}
                <div className="flex items-center justify-center">
                    <div className="w-32 h-32 bg-gray-100 rounded-lg flex items-center justify-center">
                        <span className="text-2xl font-bold text-gray-400">LOGO</span>
                    </div>
                </div>

                {/* RIGHT: Legal & Fiscal Info */}
                <div className="space-y-1 text-right">
                    <p className="text-xs text-gray-600">
                        <span className="font-semibold">Ident. Fiscal:</span> {companyInfo.fiscalId}
                    </p>
                    <p className="text-xs text-gray-600">
                        <span className="font-semibold">RC:</span> {companyInfo.rc}
                    </p>
                    <p className="text-xs text-gray-600 mt-2">
                        <span className="font-semibold">Date:</span> {formatDate(quotation.createdAt)}
                    </p>
                    <div className="mt-4 p-2 bg-blue-50 border border-blue-200 rounded">
                        <p className="text-sm font-bold text-blue-900">
                            Devis Nº {quotation.quotationNumber}
                        </p>
                    </div>
                </div>
            </div>

            {/* CLIENT INFORMATION BLOCK */}
            <div className="border-2 border-gray-300 rounded p-4 mb-6">
                <div className="grid grid-cols-2 gap-x-8 gap-y-2">
                    <div>
                        <span className="font-semibold text-sm">Client :</span>
                        <span className="text-sm ml-2">
                            {quotation.user.firstName} {quotation.user.lastName}
                            {quotation.user.companyName && ` - ${quotation.user.companyName}`}
                        </span>
                    </div>
                    <div>
                        <span className="font-semibold text-sm">Ident. Fisc. :</span>
                        <span className="text-sm ml-2">{quotation.user.taxId || 'N/A'}</span>
                    </div>
                    <div className="col-span-2">
                        <span className="font-semibold text-sm">Adresse :</span>
                        <span className="text-sm ml-2">{quotation.shippingAddress || 'N/A'}</span>
                    </div>
                    <div>
                        <span className="font-semibold text-sm">Chauf / Livr. :</span>
                        <span className="text-sm ml-2">N/A</span>
                    </div>
                    <div>
                        <span className="font-semibold text-sm">Véhicule :</span>
                        <span className="text-sm ml-2">N/A</span>
                    </div>
                </div>
            </div>

            {/* PRODUCTS TABLE */}
            <div className="mb-6">
                <table className="w-full border-collapse border border-gray-300 text-xs">
                    <thead>
                        <tr className="bg-gray-100">
                            <th className="border border-gray-300 p-2 text-left font-bold">Code</th>
                            <th className="border border-gray-300 p-2 text-left font-bold">Article / Désignation</th>
                            <th className="border border-gray-300 p-2 text-center font-bold">Unité</th>
                            <th className="border border-gray-300 p-2 text-right font-bold">Qté</th>
                            <th className="border border-gray-300 p-2 text-right font-bold">Prix Unit HT</th>
                            <th className="border border-gray-300 p-2 text-right font-bold">Remise</th>
                            <th className="border border-gray-300 p-2 text-right font-bold">Total HT</th>
                            <th className="border border-gray-300 p-2 text-right font-bold">TVA</th>
                            <th className="border border-gray-300 p-2 text-right font-bold">Total TTC</th>
                        </tr>
                    </thead>
                    <tbody>
                        {quotation.items.map((item: any, index: number) => (
                            <tr key={index} className={index % 2 === 0 ? 'bg-white' : 'bg-gray-50'}>
                                <td className="border border-gray-300 p-2">{item.productSku || 'N/A'}</td>
                                <td className="border border-gray-300 p-2">{item.productName}</td>
                                <td className="border border-gray-300 p-2 text-center">{item.selectedUnitType || 'pcs'}</td>
                                <td className="border border-gray-300 p-2 text-right">{item.quantity}</td>
                                <td className="border border-gray-300 p-2 text-right">{formatPrice(Number(item.unitPrice))}</td>
                                <td className="border border-gray-300 p-2 text-right">
                                    {Number(item.discount) > 0 ? formatPrice(Number(item.discount)) : '0.000'}
                                </td>
                                <td className="border border-gray-300 p-2 text-right font-semibold">
                                    {formatPrice(Number(item.totalHT))}
                                </td>
                                <td className="border border-gray-300 p-2 text-right">
                                    {item.tvaExempt ? 'Exonéré' : `${Number(item.tvaRate).toFixed(1)} %`}
                                </td>
                                <td className="border border-gray-300 p-2 text-right font-semibold">
                                    {formatPrice(Number(item.totalTTC))}
                                </td>
                            </tr>
                        ))}
                    </tbody>
                </table>
            </div>

            {/* SUMMARY SECTION */}
            <div className="flex justify-end mb-6">
                <div className="w-80 space-y-2 text-sm">
                    <div className="flex justify-between py-1 border-b border-gray-200">
                        <span className="text-gray-700">Total Brut HT</span>
                        <span className="font-semibold">{formatPrice(Number(quotation.totalBrutHT))} TND</span>
                    </div>
                    <div className="flex justify-between py-1 border-b border-gray-200">
                        <span className="text-gray-700">Total Remise</span>
                        <span className="font-semibold text-red-600">
                            -{formatPrice(Number(quotation.totalDiscount))} TND
                        </span>
                    </div>
                    <div className="flex justify-between py-1 border-b border-gray-200">
                        <span className="text-gray-700">Total Net HT</span>
                        <span className="font-semibold">{formatPrice(Number(quotation.totalNetHT))} TND</span>
                    </div>
                    <div className="flex justify-between py-1 border-b border-gray-200">
                        <span className="text-gray-700">Total TVA</span>
                        <span className="font-semibold">{formatPrice(Number(quotation.totalTVA))} TND</span>
                    </div>
                    <div className="flex justify-between py-2 border-t-2 border-gray-800 bg-gray-100 px-2">
                        <span className="text-gray-900 font-bold text-base">NET À PAYER</span>
                        <span className="font-bold text-base">{formatPrice(Number(quotation.netAPayer))} TND</span>
                    </div>
                </div>
            </div>

            {/* TVA DETAIL TABLE */}
            <div className="mb-6">
                <h3 className="text-sm font-bold mb-2">Détail TVA</h3>
                <table className="w-full border-collapse border border-gray-300 text-xs">
                    <thead>
                        <tr className="bg-gray-100">
                            <th className="border border-gray-300 p-2 text-left font-bold">Libellé</th>
                            <th className="border border-gray-300 p-2 text-right font-bold">Base</th>
                            <th className="border border-gray-300 p-2 text-right font-bold">Taux</th>
                            <th className="border border-gray-300 p-2 text-right font-bold">Montant</th>
                        </tr>
                    </thead>
                    <tbody>
                        {Number(quotation.tva19Amount) > 0 && (
                            <tr>
                                <td className="border border-gray-300 p-2">TVA 19%</td>
                                <td className="border border-gray-300 p-2 text-right">
                                    {formatPrice(Number(quotation.tva19Amount) / 0.19)}
                                </td>
                                <td className="border border-gray-300 p-2 text-right">19.0 %</td>
                                <td className="border border-gray-300 p-2 text-right font-semibold">
                                    {formatPrice(Number(quotation.tva19Amount))}
                                </td>
                            </tr>
                        )}
                        {Number(quotation.tva7Amount) > 0 && (
                            <tr>
                                <td className="border border-gray-300 p-2">TVA 7%</td>
                                <td className="border border-gray-300 p-2 text-right">
                                    {formatPrice(Number(quotation.tva7Amount) / 0.07)}
                                </td>
                                <td className="border border-gray-300 p-2 text-right">7.0 %</td>
                                <td className="border border-gray-300 p-2 text-right font-semibold">
                                    {formatPrice(Number(quotation.tva7Amount))}
                                </td>
                            </tr>
                        )}
                        {Number(quotation.tva0Amount) === 0 && Number(quotation.totalTVA) === 0 && (
                            <tr>
                                <td className="border border-gray-300 p-2">Exonéré</td>
                                <td className="border border-gray-300 p-2 text-right">
                                    {formatPrice(Number(quotation.totalNetHT))}
                                </td>
                                <td className="border border-gray-300 p-2 text-right">0.0 %</td>
                                <td className="border border-gray-300 p-2 text-right font-semibold">0.000</td>
                            </tr>
                        )}
                        <tr className="bg-gray-100 font-bold">
                            <td className="border border-gray-300 p-2">TOTAL</td>
                            <td className="border border-gray-300 p-2 text-right">
                                {formatPrice(Number(quotation.totalNetHT))}
                            </td>
                            <td className="border border-gray-300 p-2 text-right">-</td>
                            <td className="border border-gray-300 p-2 text-right">
                                {formatPrice(Number(quotation.totalTVA))}
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>

            {/* FOOTER - Signatures */}
            <div className="mt-8 pt-4 border-t border-gray-300">
                <div className="grid grid-cols-3 gap-8 mb-4">
                    <div className="text-center">
                        <p className="text-xs font-semibold mb-8">Le Magasinier</p>
                        <div className="border-t border-gray-400 pt-1"></div>
                    </div>
                    <div className="text-center">
                        <p className="text-xs font-semibold mb-8">Service Commercial</p>
                        <div className="border-t border-gray-400 pt-1"></div>
                    </div>
                    <div className="text-center">
                        <p className="text-xs font-semibold mb-8">Décharge Client</p>
                        <div className="border-t border-gray-400 pt-1"></div>
                    </div>
                </div>
                <p className="text-xs text-center text-gray-600 mb-2">
                    Vous avez été servi par : Service Commercial
                </p>
                <div className="bg-gray-200 p-2 text-center">
                    <p className="text-xs text-gray-700">
                        {companyInfo.address}, {companyInfo.city} {companyInfo.postalCode} -
                        Tél: {companyInfo.phones.join(' / ')} - Email: {companyInfo.email}
                    </p>
                </div>
            </div>

            {/* Print Button */}
            <div className="mt-6 text-center no-print">
                <button
                    onClick={() => window.print()}
                    className="bg-blue-600 text-white px-6 py-2 rounded hover:bg-blue-700 transition-colors"
                >
                    Imprimer / Télécharger PDF
                </button>
            </div>

            <style>{`
        @media print {
          .no-print {
            display: none;
          }
          body {
            margin: 0;
            padding: 0;
          }
        }
      `}</style>
        </div>
    );
}
