import prisma from '../config/database';
import {
    PrismaClient,
    Prisma,
    StockTransactionType,
    StockLocationType
} from '@prisma/client';

// Define simplified interfaces if types are hard to import directly in this mixed env
// OR import them from the namespace if available, but usually they are exported types.

// WORKAROUND: Import types as type-only import
import type {
    SupplierInventory,
    CustomerInventory,
    StockTransaction,
    Product,
    OrderItem
} from '@prisma/client';

// Enums are now imported top-level above

export interface InventoryReport {
    totalItems: number;
    totalValue: number;
    lowStockItems?: number;
}

export interface TransferResult {
    success: boolean;
    message: string;
    transactions?: StockTransaction[];
    errors?: string[];
}

export interface ValidationResult {
    isValid: boolean;
    outOfStockItems: {
        productId: string;
        productName: string;
        requested: number;
        available: number;
    }[];
}

export class InventoryService {

    // ==========================================
    // SUPPLIER INVENTORY
    // ==========================================

    /**
     * Get current stock level for a product
     */
    async checkSupplierStock(productId: string): Promise<number> {
        const inventory = await prisma.supplierInventory.findUnique({
            where: { productId }
        });
        return inventory?.quantity || 0;
    }

    /**
   * Add stock to supplier inventory (Purchase, Adjustment, Returns)
   */
    async addSupplierStock(
        productId: string,
        quantity: number,
        notes: string = 'Manual Adjustment',
        type: any = 'ADJUSTMENT' // Use string literal to avoid runtime Enum lookup issues
    ): Promise<SupplierInventory> {
        return await prisma.$transaction(async (tx) => {
            // 1. Update or Create Inventory Record
            const inventory = await tx.supplierInventory.upsert({
                where: { productId },
                create: {
                    productId,
                    quantity,
                    lastRestockDate: new Date()
                },
                update: {
                    quantity: { increment: quantity },
                    lastRestockDate: new Date()
                }
            });

            // 2. Create Audit Log
            await tx.stockTransaction.create({
                data: {
                    productId,
                    type,
                    quantity,
                    toLocationType: 'SUPPLIER' as any,
                    notes
                }
            });

            // 3. Update Legacy Product Stock Field (Backward Compatibility)
            await tx.product.update({
                where: { id: productId },
                data: { stockQuantity: inventory.quantity }
            });

            return inventory;
        });
    }

    /**
     * Deduct stock from supplier inventory
     */
    async deductSupplierStock(
        productId: string,
        quantity: number,
        referenceId?: string,
        notes: string = 'Deduction'
    ): Promise<SupplierInventory> {
        return await prisma.$transaction(async (tx) => {
            // 1. Check stock
            const current = await tx.supplierInventory.findUnique({ where: { productId } });
            if (!current || current.quantity < quantity) {
                throw new Error(`Insufficient stock for product ${productId}. Requested: ${quantity}, Available: ${current?.quantity || 0}`);
            }

            // 2. Deduct
            const inventory = await tx.supplierInventory.update({
                where: { productId },
                data: { quantity: { decrement: quantity } }
            });

            // 3. Audit Log
            await tx.stockTransaction.create({
                data: {
                    productId,
                    type: 'SALE' as any,
                    quantity: -quantity,
                    fromLocationType: 'SUPPLIER' as any,
                    referenceId, // Order ID usually
                    notes
                }
            });

            // 4. Update Legacy Product Stock Field
            await tx.product.update({
                where: { id: productId },
                data: { stockQuantity: inventory.quantity }
            });

            return inventory;
        });
    }

    /**
     * Get products below reorder level
     */
    async getLowStockProducts(threshold?: number): Promise<any[]> {
        // If threshold provided, use it, else use product's own reorderLevel
        const whereClause: Prisma.SupplierInventoryWhereInput = threshold !== undefined
            ? { quantity: { lte: threshold } }
            : { quantity: { lte: prisma.supplierInventory.fields.reorderLevel } };

        return await prisma.supplierInventory.findMany({
            where: whereClause,
            include: { product: true }
        });
    }

    // ==========================================
    // CUSTOMER INVENTORY
    // ==========================================

    /**
     * Get customer's inventory
     */
    async getCustomerInventory(customerId: string): Promise<CustomerInventory[]> {
        return await prisma.customerInventory.findMany({
            where: { customerId },
            include: { product: true, sourceOrder: true }
        });
    }

    /**
     * Add to customer inventory (usually from order delivery)
     */
    async addToCustomerInventory(
        customerId: string,
        productId: string,
        quantity: number,
        sourceOrderId: string
    ): Promise<CustomerInventory> {
        return await prisma.$transaction(async (tx) => {
            // Create new entry for each receipt to track source order
            // We could aggregate by product, but requirements say "received date" and "source order" are important
            // So let's keep separate entries or aggregated?
            // "View their inventory dashboard... track received orders"
            // Suggestion: CustomerInventory table seems designed to hold specific receipt batches or aggregated?
            // Schema has sourceOrderId. If we want aggregate view, we query sum.
            // Let's create a new record for this specific receipt logic or update if same order/product combo exists?

            const inventory = await tx.customerInventory.create({
                data: {
                    customerId,
                    productId,
                    quantity,
                    sourceOrderId,
                    receivedAt: new Date()
                }
            });

            await tx.stockTransaction.create({
                data: {
                    productId,
                    type: 'TRANSFER' as any,
                    quantity,
                    toLocationType: 'CUSTOMER' as any,
                    toLocationId: customerId,
                    referenceId: sourceOrderId,
                    notes: 'Received from Order'
                }
            });

            return inventory;
        });
    }

    // ==========================================
    // STOCK TRANSFER (ORDER FULFILLMENT)
    // ==========================================

    /**
     * Validates if we can fulfill an order
     */
    async validateOrderStock(orderItems: OrderItem[]): Promise<ValidationResult> {
        const outOfStockItems = [];

        for (const item of orderItems) {
            const stock = await this.checkSupplierStock(item.productId);
            if (stock < item.quantity) {
                outOfStockItems.push({
                    productId: item.productId,
                    productName: item.productName,
                    requested: item.quantity,
                    available: stock
                });
            }
        }

        return {
            isValid: outOfStockItems.length === 0,
            outOfStockItems
        };
    }

    /**
     * CORE: Process stock transfer for an Order
     * Deducts from Supplier -> Adds to Customer
     */
    async processOrderStockTransfer(orderId: string): Promise<TransferResult> {
        const order = await prisma.order.findUnique({
            where: { id: orderId },
            include: { items: true }
        });

        if (!order) throw new Error('Order not found');
        if (!order.userId) throw new Error('Order has no customer assigned');

        // 1. Validate first
        const validation = await this.validateOrderStock(order.items);
        if (!validation.isValid) {
            return {
                success: false,
                message: 'Insufficient stock',
                errors: validation.outOfStockItems.map(i => `${i.productName}: Need ${i.requested}, have ${i.available}`)
            };
        }

        try {
            await prisma.$transaction(async (tx) => {
                for (const item of order.items) {
                    if (item.inventoryProcessed) continue; // Skip if already processed

                    // A. Deduct from Supplier
                    // Verify again inside transaction for safety
                    const supplierStock = await tx.supplierInventory.findUnique({ where: { productId: item.productId } });
                    if (!supplierStock || supplierStock.quantity < item.quantity) {
                        throw new Error(`Stock changed during processing for ${item.productName}`);
                    }

                    await tx.supplierInventory.update({
                        where: { productId: item.productId },
                        data: { quantity: { decrement: item.quantity } }
                    });

                    // Sync legacy field
                    await tx.product.update({
                        where: { id: item.productId },
                        data: { stockQuantity: { decrement: item.quantity } }
                    });

                    // B. Add to Customer
                    await tx.customerInventory.create({
                        data: {
                            customerId: order.userId!,
                            productId: item.productId,
                            quantity: item.quantity,
                            sourceOrderId: order.id
                        }
                    });

                    // C. Log Transaction (Transfer)
                    await tx.stockTransaction.create({
                        data: {
                            productId: item.productId,
                            type: 'TRANSFER' as any,
                            quantity: item.quantity,
                            fromLocationType: 'SUPPLIER' as any,
                            toLocationType: 'CUSTOMER' as any,
                            toLocationId: order.userId!,
                            referenceId: order.id,
                            notes: `Order fulfillment ${order.orderNumber}`
                        }
                    });

                    // D. Mark item as processed
                    await tx.orderItem.update({
                        where: { id: item.id },
                        data: { inventoryProcessed: true, stockTransactionId: 'BATCH' } // Ideal: store specific transaction ID
                    });
                }
            });

            return { success: true, message: 'Stock transferred successfully' };

        } catch (error: any) {
            console.error('Stock transfer failed:', error);
            return { success: false, message: error.message };
        }
    }

    /**
     * Get transaction history
     */
    async getStockTransactionHistory(filters: any) {
        return await prisma.stockTransaction.findMany({
            where: filters, // Need to map specific filters strictly
            include: { product: true },
            orderBy: { createdAt: 'desc' }
        });
    }
}

export const inventoryService = new InventoryService();
