import { PrismaClient } from '@prisma/client';
const prisma = new PrismaClient();
/**
 * Tunisian TVA rates (as percentages)
 */
export const TVA_RATES = {
    STANDARD: 19, // Standard rate (taux normal)
    REDUCED: 7, // Reduced rate (taux réduit) - basic necessities, tourism
    EXEMPT: 0, // Exempt products (exonéré)
};
/**
 * Get effective TVA rate for a product in a specific catalog
 * Checks catalog-specific overrides first, then falls back to product default
 */
export async function getEffectiveTvaRate(productId, catalogId) {
    // 1. Check for catalog-specific override
    const catalogOverride = await prisma.catalogProductTva.findUnique({
        where: {
            catalogId_productId: {
                catalogId,
                productId,
            },
        },
    });
    if (catalogOverride) {
        return {
            rate: Number(catalogOverride.tvaRate),
            exempt: catalogOverride.tvaExempt,
            reason: catalogOverride.reason || undefined,
        };
    }
    // 2. Use product default TVA
    const product = await prisma.product.findUnique({
        where: { id: productId },
        select: { tvaRate: true, tvaExempt: true, tvaExemptReason: true },
    });
    if (!product) {
        throw new Error(`Product ${productId} not found`);
    }
    return {
        rate: Number(product.tvaRate),
        exempt: product.tvaExempt,
        reason: product.tvaExemptReason || undefined,
    };
}
/**
 * Calculate TVA for a single line item
 */
export function calculateLineItemTva(item) {
    const totalBrutHT = item.quantity * item.unitPrice;
    const discount = item.discount || 0;
    const totalNetHT = totalBrutHT - discount;
    // FODEC Calculation (1% of Net HT)
    const fodecAmount = item.fodecSubject ? (totalNetHT * 0.01) : 0;
    // TVA Base = Net HT + FODEC
    const tvaBase = totalNetHT + fodecAmount;
    const tvaAmount = item.tvaExempt ? 0 : (tvaBase * item.tvaRate) / 100;
    const totalTTC = totalNetHT + fodecAmount + tvaAmount;
    return {
        totalBrutHT: roundTo3Decimals(totalBrutHT),
        discount: roundTo3Decimals(discount),
        totalNetHT: roundTo3Decimals(totalNetHT),
        fodecAmount: roundTo3Decimals(fodecAmount),
        tvaBase: roundTo3Decimals(tvaBase),
        tvaRate: item.tvaRate,
        tvaAmount: roundTo3Decimals(tvaAmount),
        totalTTC: roundTo3Decimals(totalTTC),
        tvaExempt: item.tvaExempt,
    };
}
export function calculateDocumentTva(lineItems) {
    let totalBrutHT = 0;
    let totalDiscount = 0;
    let totalNetHT = 0;
    let totalFodec = 0;
    let tva19Amount = 0;
    let tva7Amount = 0;
    let tva0Amount = 0;
    for (const line of lineItems) {
        totalBrutHT += line.totalBrutHT;
        totalDiscount += line.discount;
        totalNetHT += line.totalNetHT;
        totalFodec += line.fodecAmount || 0;
        // Categorize TVA by rate
        if (line.tvaRate === 19) {
            tva19Amount += line.tvaAmount;
        }
        else if (line.tvaRate === 7) {
            tva7Amount += line.tvaAmount;
        }
        else {
            tva0Amount += line.tvaAmount;
        }
    }
    const totalTVA = tva19Amount + tva7Amount + tva0Amount;
    const netAPayer = totalNetHT + totalFodec + totalTVA; // Include FODEC in final sum
    return {
        totalBrutHT: roundTo3Decimals(totalBrutHT),
        totalDiscount: roundTo3Decimals(totalDiscount),
        totalNetHT: roundTo3Decimals(totalNetHT),
        totalFodec: roundTo3Decimals(totalFodec),
        tva19Amount: roundTo3Decimals(tva19Amount),
        tva7Amount: roundTo3Decimals(tva7Amount),
        tva0Amount: roundTo3Decimals(tva0Amount),
        totalTVA: roundTo3Decimals(totalTVA),
        netAPayer: roundTo3Decimals(netAPayer),
        netAPayerWords: numberToFrenchWords(roundTo3Decimals(netAPayer)),
    };
}
/**
 * Helper to round to 3 decimals (Millimes)
 */
function roundTo3Decimals(num) {
    return Math.round(num * 1000) / 1000;
}
/**
 * Convert number to French words for invoices
 * Example: 1234.56 → "Mille deux cent trente-quatre dinars et cinquante-six millimes"
 */
export function numberToFrenchWords(amount) {
    const dinars = Math.floor(amount);
    const millimes = Math.round((amount - dinars) * 1000);
    const units = ['', 'un', 'deux', 'trois', 'quatre', 'cinq', 'six', 'sept', 'huit', 'neuf'];
    const teens = ['dix', 'onze', 'douze', 'treize', 'quatorze', 'quinze', 'seize', 'dix-sept', 'dix-huit', 'dix-neuf'];
    const tens = ['', 'dix', 'vingt', 'trente', 'quarante', 'cinquante', 'soixante', 'soixante-dix', 'quatre-vingt', 'quatre-vingt-dix'];
    function convertLessThanThousand(n) {
        if (n === 0)
            return '';
        if (n < 10)
            return units[n];
        if (n < 20)
            return teens[n - 10];
        if (n < 100) {
            const ten = Math.floor(n / 10);
            const unit = n % 10;
            if (ten === 7 || ten === 9) {
                const base = ten === 7 ? 6 : 8;
                return tens[base] + '-' + (unit === 0 ? 'dix' : teens[unit]);
            }
            return tens[ten] + (unit ? '-' + units[unit] : '');
        }
        const hundred = Math.floor(n / 100);
        const rest = n % 100;
        let result = hundred === 1 ? 'cent' : units[hundred] + ' cent';
        if (hundred > 1 && rest === 0)
            result += 's';
        return result + (rest ? ' ' + convertLessThanThousand(rest) : '');
    }
    function convertNumber(n) {
        if (n === 0)
            return 'zéro';
        if (n < 1000)
            return convertLessThanThousand(n);
        const thousands = Math.floor(n / 1000);
        const rest = n % 1000;
        let result = '';
        if (thousands === 1) {
            result = 'mille';
        }
        else {
            result = convertLessThanThousand(thousands) + ' mille';
        }
        if (rest) {
            result += ' ' + convertLessThanThousand(rest);
        }
        return result;
    }
    let result = convertNumber(dinars);
    if (result) {
        result = result.charAt(0).toUpperCase() + result.slice(1);
        result += dinars > 1 ? ' dinars' : ' dinar';
    }
    if (millimes > 0) {
        result += ' et ' + convertNumber(millimes);
        result += millimes > 1 ? ' millimes' : ' millime';
    }
    return result || 'Zéro dinar';
}
/**
 * Check if customer is TVA exempt
 */
export async function isCustomerTvaExempt(userId) {
    const user = await prisma.user.findUnique({
        where: { id: userId },
        select: {
            tvaExempt: true,
            tvaExemptValidUntil: true
        },
    });
    if (!user || !user.tvaExempt)
        return false;
    // Check if exemption is still valid
    if (user.tvaExemptValidUntil && user.tvaExemptValidUntil < new Date()) {
        return false;
    }
    return true;
}
/**
 * Format amount for display
 */
export function formatTvaAmount(amount) {
    return amount.toFixed(2);
}
/**
 * Get TVA rate label in French
 */
export function getTvaRateLabel(rate) {
    if (rate === 19)
        return 'TVA 19% (Taux normal)';
    if (rate === 7)
        return 'TVA 7% (Taux réduit)';
    if (rate === 0)
        return 'Exonéré de TVA';
    return `TVA ${rate}%`;
}
//# sourceMappingURL=tvaCalculations.js.map