import { PrismaClient } from '@prisma/client';
import { inventoryService } from '../services/inventory.service.js';
// Mock req/res
const mockRequest = (body, user, params = {}) => ({
    body,
    user,
    params,
    query: {},
    header: (name) => {
        if (name === 'x-catalog-id')
            return 'CAT_TEST_123';
        return null;
    },
    get: (name) => {
        if (name === 'x-catalog-id')
            return 'CAT_TEST_123';
        return null;
    }
});
const mockResponse = () => {
    const res = {};
    res.status = (code) => {
        res.statusCode = code;
        return res;
    };
    res.json = (data) => {
        res.body = data;
        return res;
    };
    return res;
};
// We will use the controllers directly to test the flow
import * as quotationController from '../controllers/quotation.controller.js';
const prisma = new PrismaClient();
async function runVerification() {
    console.log('Starting Customer Inventory Flow Verification...');
    try {
        // 1. Setup Data
        console.log('1. Setting up test data...');
        // Create/Get Catalog
        const catalog = await prisma.catalog.upsert({
            where: { id: 'CAT_TEST_123' },
            create: {
                id: 'CAT_TEST_123',
                name: 'Test Catalog',
                slug: 'test-catalog',
                isActive: true
            },
            update: {}
        });
        // Create Category
        const category = await prisma.category.upsert({
            where: { catalogId_slug: { catalogId: catalog.id, slug: 'test-category' } },
            create: {
                catalogId: catalog.id,
                name: 'Test Category',
                slug: 'test-category',
                isActive: true
            },
            update: {}
        });
        // Create User (Customer)
        const customer = await prisma.user.upsert({
            where: { email: 'testcustomer@example.com' },
            create: {
                email: 'testcustomer@example.com',
                passwordHash: 'hashedpassword',
                firstName: 'Test',
                lastName: 'Customer',
                role: 'CUSTOMER',
                catalogAssignments: {
                    create: {
                        catalogId: catalog.id
                    }
                }
            },
            update: {},
        });
        // Create User (Admin)
        const admin = await prisma.user.findFirst({ where: { role: 'ADMIN' } }) || customer; // Fallback if no admin
        // Create Product
        const product = await prisma.product.upsert({
            where: {
                catalogId_sku: {
                    catalogId: catalog.id,
                    sku: 'TEST-SKU-INV-5'
                }
            },
            create: {
                sku: 'TEST-SKU-INV-5',
                name: 'Test Product Inventory',
                slug: 'test-product-inventory',
                basePrice: 100,
                stockQuantity: 5, // INITIAL STOCK
                catalogId: catalog.id,
                categoryId: category.id
            },
            update: {
                stockQuantity: 5 // Reset stock
            },
        });
        // Ensure Supplier Inventory matches
        await inventoryService.addSupplierStock(product.id, 5, 'Reset for Test', 'ADJUSTMENT');
        // Force set to 5 in case addSupplierStock added to existing
        await prisma.supplierInventory.update({
            where: { productId: product.id },
            data: { quantity: 5 }
        });
        console.log(`   Product ${product.sku} stock reset to 5.`);
        // Create Address
        const address = await prisma.address.create({
            data: {
                userId: customer.id,
                street: '123 Test St',
                city: 'Test City',
                country: 'Tunisia',
                isDefault: true
            }
        });
        // 2. Test: OOS Prevention
        console.log('\n2. Testing Out-of-Stock Prevention...');
        const reqOOS = mockRequest({
            items: [{ productId: product.id, quantity: 10 }], // Request 10, have 5
            shippingAddressId: address.id,
        }, customer);
        const resOOS = mockResponse();
        await quotationController.createQuotation(reqOOS, resOOS);
        if (resOOS.statusCode === 400 && resOOS.body.error?.message?.includes('Insufficient stock')) {
            console.log('   ✅ PASS: OOS Quotation blocked as expected.');
        }
        else {
            console.error('   ❌ FAIL: OOS Quotation was not blocked properly.');
            console.log('Response:', resOOS.statusCode, resOOS.body);
        }
        // 3. Test: Successful Quotation
        console.log('\n3. Testing Successful Quotation Creation...');
        const reqSuccess = mockRequest({
            items: [{ productId: product.id, quantity: 3 }], // Request 3, have 5
            shippingAddressId: address.id,
        }, customer);
        const resSuccess = mockResponse();
        await quotationController.createQuotation(reqSuccess, resSuccess);
        if (resSuccess.statusCode === 201 && resSuccess.body.data?.id) {
            console.log(`   ✅ PASS: Quotation created: ${resSuccess.body.data.quotationNumber}`);
        }
        else {
            console.error('   ❌ FAIL: Quotation creation failed.');
            console.log('Response:', resSuccess.statusCode, resSuccess.body);
            return;
        }
        const quotationId = resSuccess.body.data.id;
        // 4. Test: Submit Quotation
        console.log('\n4. Submitting Quotation...');
        const reqSubmit = mockRequest({}, customer, { id: quotationId });
        const resSubmit = mockResponse();
        await quotationController.submitQuotation(reqSubmit, resSubmit);
        console.log('   Quotation submitted status:', resSubmit.body.data.status);
        // 5. Test: Approve & Stock Transfer
        console.log('\n5. Testing Approval & Stock Transfer...');
        const reqApprove = mockRequest({}, admin, { id: quotationId });
        const resApprove = mockResponse();
        await quotationController.approveQuotation(reqApprove, resApprove);
        if (resApprove.body?.success) {
            console.log('   ✅ PASS: Quotation Approved & Converted to Order.');
            const orderId = resApprove.body.data.order.id;
            console.log(`   Order ID: ${orderId}`);
            // Verify Stock Levels
            const updatedProduct = await prisma.product.findUnique({ where: { id: product.id } });
            const supplierStock = await inventoryService.checkSupplierStock(product.id);
            const customerStock = await inventoryService.getCustomerInventory(customer.id);
            const transaction = await prisma.stockTransaction.findFirst({
                where: { referenceId: orderId, type: 'TRANSFER' },
                orderBy: { createdAt: 'desc' }
            });
            console.log(`   Expected Stock: 5 - 3 = 2`);
            console.log(`   Actual Product Stock (Legacy): ${updatedProduct?.stockQuantity}`);
            console.log(`   Actual Supplier Stock: ${supplierStock}`);
            if (supplierStock === 2) {
                console.log('   ✅ PASS: Stock deducted correctly.');
            }
            else {
                console.error('   ❌ FAIL: Stock deduction incorrect.');
            }
            const receivedItem = customerStock.find(i => i.sourceOrderId === orderId && i.productId === product.id);
            if (receivedItem && receivedItem.quantity === 3) {
                console.log('   ✅ PASS: Customer Inventory updated correctly (Received 3).');
            }
            else {
                console.error('   ❌ FAIL: Customer Inventory not updated.', customerStock);
            }
            if (transaction) {
                console.log('   ✅ PASS: Stock Transaction logged.');
            }
            else {
                console.error('   ❌ FAIL: No Stock Transaction log found.');
            }
        }
        else {
            console.error('   ❌ FAIL: Approval failed.');
            console.log(resApprove.body);
        }
    }
    catch (error) {
        console.error('Test Verification Failed:', error);
    }
    finally {
        await prisma.$disconnect();
        console.log('\nVerification Cleaned Up.');
    }
}
runVerification();
//# sourceMappingURL=verify-customer-flow.js.map