import jwt from 'jsonwebtoken';
import { createError } from './errorHandler.js';
export const ROLES = {
    SUPER_ADMIN: 'SUPER_ADMIN',
    ADMIN: 'ADMIN', // Legacy, treated as Super Admin
    SALES_ADMIN: 'SALES_ADMIN',
    CATALOG_ADMIN: 'CATALOG_ADMIN',
    ACCOUNTING_ADMIN: 'ACCOUNTING_ADMIN',
    SUPPORT_ADMIN: 'SUPPORT_ADMIN',
    MANAGER: 'MANAGER',
    CUSTOMER: 'CUSTOMER'
};
export const PERMISSIONS = {
    // Groups
    ALL_ADMINS: ['SUPER_ADMIN', 'ADMIN', 'SALES_ADMIN', 'CATALOG_ADMIN', 'ACCOUNTING_ADMIN', 'SUPPORT_ADMIN', 'MANAGER'],
    FULL_ACCESS: ['SUPER_ADMIN', 'ADMIN'],
    CATALOG_MANAGE: ['SUPER_ADMIN', 'ADMIN', 'CATALOG_ADMIN'],
    SALES_MANAGE: ['SUPER_ADMIN', 'ADMIN', 'SALES_ADMIN'],
    SALES_VIEW: ['SUPER_ADMIN', 'ADMIN', 'SALES_ADMIN', 'SUPPORT_ADMIN'], // Support can view
    ACCOUNTING_MANAGE: ['SUPER_ADMIN', 'ADMIN', 'ACCOUNTING_ADMIN'], // Re-check if Accounting is read-only? User said "Read-only access to accounting". But typically they might need to update status. I will assume Manage for now for "Accounting Admin" unless specified strictly read-only for EVERYTHING. Actually "Accounting Admin: Read-only access to accounting, invoices...". okay, I will make a ACCOUNTING_VIEW group.
    ACCOUNTING_VIEW: ['SUPER_ADMIN', 'ADMIN', 'ACCOUNTING_ADMIN', 'SALES_ADMIN'], // Sales can see invoices? "Sales Admin: ... invoices". Yes.
    CUSTOMER_MANAGE: ['SUPER_ADMIN', 'ADMIN', 'SALES_ADMIN'],
    CUSTOMER_VIEW: ['SUPER_ADMIN', 'ADMIN', 'SALES_ADMIN', 'SUPPORT_ADMIN'],
    SETTINGS_MANAGE: ['SUPER_ADMIN', 'ADMIN'],
};
export const authenticate = async (req, res, next) => {
    try {
        const authHeader = req.headers.authorization;
        if (!authHeader || !authHeader.startsWith('Bearer ')) {
            throw createError('No token provided', 401);
        }
        const token = authHeader.substring(7);
        const secret = process.env.JWT_SECRET;
        if (!secret) {
            throw createError('JWT secret not configured', 500);
        }
        const decoded = jwt.verify(token, secret);
        req.user = decoded;
        next();
    }
    catch (error) {
        if (error instanceof jwt.JsonWebTokenError) {
            next(createError('Invalid token', 401));
        }
        else if (error instanceof jwt.TokenExpiredError) {
            next(createError('Token expired', 401));
        }
        else {
            next(error);
        }
    }
};
export const authorize = (...roles) => {
    return (req, res, next) => {
        if (!req.user) {
            return next(createError('Not authenticated', 401));
        }
        if (!roles.includes(req.user.role)) {
            return next(createError('Not authorized to access this resource', 403));
        }
        next();
    };
};
//# sourceMappingURL=auth.js.map