import bcrypt from 'bcryptjs';
import prisma from '../config/database.js';
import { createError } from '../middleware/errorHandler.js';
import { UserRole } from '@prisma/client';
// List all admins
export const listAdmins = async (req, res, next) => {
    try {
        const admins = await prisma.user.findMany({
            where: {
                role: {
                    in: [
                        'ADMIN',
                        'SUPER_ADMIN',
                        'SALES_ADMIN',
                        'CATALOG_ADMIN',
                        'ACCOUNTING_ADMIN',
                        'SUPPORT_ADMIN',
                        'MANAGER'
                    ]
                }
            },
            select: {
                id: true,
                email: true,
                username: true,
                firstName: true,
                lastName: true,
                role: true,
                isActive: true,
                createdAt: true
            },
            orderBy: { createdAt: 'desc' }
        });
        res.json({
            success: true,
            data: admins
        });
    }
    catch (error) {
        next(error);
    }
};
// Create new admin
export const createAdmin = async (req, res, next) => {
    try {
        const { email, password, firstName, lastName, role, phone } = req.body;
        if (!email || !password || !role) {
            throw createError('Email, password, and role are required', 400);
        }
        if (!Object.values(UserRole).includes(role)) {
            throw createError('Invalid role', 400);
        }
        const existingUser = await prisma.user.findUnique({
            where: { email }
        });
        if (existingUser) {
            throw createError('User already exists', 409);
        }
        const passwordHash = await bcrypt.hash(password, 10);
        const admin = await prisma.user.create({
            data: {
                email,
                passwordHash,
                firstName,
                lastName,
                phone,
                role: role,
                isActive: true
            },
            select: {
                id: true,
                email: true,
                role: true,
                isActive: true
            }
        });
        res.status(201).json({
            success: true,
            data: admin
        });
    }
    catch (error) {
        next(error);
    }
};
// Update admin role
export const updateAdminRole = async (req, res, next) => {
    try {
        const { id } = req.params;
        const { role } = req.body;
        if (!role || !Object.values(UserRole).includes(role)) {
            throw createError('Valid role is required', 400);
        }
        // Prevent changing own role to avoid lockout (optional safety)
        if (req.user?.id === id) {
            // Allow if super admin? Maybe restrict self-demotion
        }
        const admin = await prisma.user.update({
            where: { id },
            data: { role: role },
            select: {
                id: true,
                email: true,
                role: true
            }
        });
        res.json({
            success: true,
            data: admin
        });
    }
    catch (error) {
        next(error);
    }
};
// Delete/Deactivate admin
export const deleteAdmin = async (req, res, next) => {
    try {
        const { id } = req.params;
        if (req.user?.id === id) {
            throw createError('Cannot delete yourself', 400);
        }
        await prisma.user.delete({
            where: { id }
        });
        res.json({
            success: true,
            message: 'Admin deleted successfully'
        });
    }
    catch (error) {
        next(error);
    }
};
//# sourceMappingURL=admin.controller.js.map