/**
 * Currency Utility Functions
 * Standardizes currency formatting for the Admin Dashboard
 * Default: Tunisian Dinar (TND) with 3 decimal places
 */

export const CURRENCY_CODE = 'TND';
export const CURRENCY_DECIMALS = 3;

/**
 * Format a number as TND currency with 3 decimal places
 * @param amount - The amount to format
 * @returns Formatted string (e.g., "123.456 TND")
 */
export function formatPrice(amount: number | string): string {
    const numAmount = typeof amount === 'string' ? parseFloat(amount) : amount;

    if (isNaN(numAmount)) {
        return `0.000 ${CURRENCY_CODE}`;
    }

    return `${numAmount.toFixed(CURRENCY_DECIMALS)} ${CURRENCY_CODE}`;
}

/**
 * Format a number as TND currency without the currency code
 * @param amount - The amount to format
 * @returns Formatted string (e.g., "123.456")
 */
export function formatAmount(amount: number | string): string {
    const numAmount = typeof amount === 'string' ? parseFloat(amount) : amount;

    if (isNaN(numAmount)) {
        return '0.000';
    }

    return numAmount.toFixed(CURRENCY_DECIMALS);
}

/**
 * Format for input fields (same as formatAmount for now)
 */
export function formatForInput(amount: number | string): string {
    return formatAmount(amount);
}
