import { buildAuthHeaders } from '../utils/apiClient';

const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:3001/api';

export interface Plugin {
    name: string;
    version: string;
    status: 'ACTIVE' | 'INACTIVE';
    description?: string;
}

export const pluginService = {
    getPlugins: async (): Promise<Plugin[]> => {
        const response = await fetch(`${API_URL}/admin/plugins`, {
            headers: buildAuthHeaders()
        });

        if (!response.ok) {
            throw new Error('Failed to fetch plugins');
        }

        const data = await response.json();
        return data.data;
    },

    togglePlugin: async (name: string): Promise<void> => {
        const response = await fetch(`${API_URL}/admin/plugins/${name}/toggle`, {
            method: 'POST',
            headers: buildAuthHeaders()
        });

        if (!response.ok) {
            throw new Error('Failed to toggle plugin');
        }
    },

    getNavigation: async (): Promise<any[]> => {
        const response = await fetch(`${API_URL}/admin/plugins/navigation`, {
            headers: buildAuthHeaders()
        });
        if (!response.ok) return [];
        const data = await response.json();
        return data.data;
    },

    uploadPlugin: async (file: File): Promise<void> => {
        const formData = new FormData();
        formData.append('plugin', file);

        const response = await fetch(`${API_URL}/admin/plugins/upload`, {
            method: 'POST',
            // Note: Do NOT set Content-Type header when using FormData; fetch does it automatically with boundary
            headers: {
                ...buildAuthHeaders({ includeJson: false }) // helper update needed or manual header
            },
            body: formData
        });

        if (!response.ok) {
            const err = await response.json();
            throw new Error(err.error || 'Failed to upload plugin');
        }
    }
};
