# Multi-Catalog Testing Guide

## Overview

This guide helps you test the multi-catalog feature in a safe local VirtualBox environment before deploying to production.

## Prerequisites

- VirtualBox installed on your machine
- Ubuntu Server 22.04 ISO
- At least 4GB RAM and 20GB disk space for VM

## Step 1: Create VirtualBox VM

### 1.1 Create New VM
1. Open VirtualBox
2. Click "New"
3. Name: `SMD-Test-Server`
4. Type: Linux
5. Version: Ubuntu (64-bit)
6. RAM: 4096 MB (4GB)
7. Create virtual hard disk: 20 GB, VDI, Dynamically allocated

### 1.2 Configure Network
1. Select the VM → Settings → Network
2. Adapter 1: Enable Network Adapter
3. Attached to: **Bridged Adapter**
4. This allows you to access the VM from your host machine

### 1.3 Install Ubuntu
1. Start the VM
2. Select Ubuntu Server 22.04 ISO
3. Follow installation wizard
4. Create user (e.g., `testuser`)
5. Install OpenSSH server when prompted

### 1.4 Get VM IP Address
After installation, login to VM and run:
```bash
ip addr show
```
Note the IP address (e.g., `192.168.1.100`)

## Step 2: Install Docker on VM

SSH into your VM from host machine:
```bash
ssh testuser@192.168.1.100
```

Install Docker and Docker Compose:
```bash
# Update system
sudo apt update && sudo apt upgrade -y

# Install Docker
sudo apt install -y docker.io docker-compose git

# Add your user to docker group
sudo usermod -aG docker $USER

# Logout and login again
exit
ssh testuser@192.168.1.100

# Verify Docker is working
docker --version
docker-compose --version
```

## Step 3: Transfer Project to VM

### Option A: Using SCP (from your host machine)

```bash
# Navigate to project directory
cd /home/cube/Documents/theSmartCatalogsmd/pwd_issue_see_withgravity/sacond_update

# Create tarball
tar -czf pwd_update_gravity.tar.gz pwd_update_gravity/

# Copy to VM (replace IP with your VM's IP)
scp pwd_update_gravity.tar.gz testuser@192.168.1.100:~/

# SSH into VM and extract
ssh testuser@192.168.1.100
tar -xzf pwd_update_gravity.tar.gz
cd pwd_update_gravity/backend
```

### Option B: Using Git (if you have a repository)

```bash
# On VM
git clone <your-repo-url>
cd pwd_update_gravity/backend
```

## Step 4: Setup Test Environment

On the VM, in the backend directory:

```bash
# Copy test environment file
cp .env.test .env

# Verify the configuration
cat .env
```

## Step 5: Start Test Environment

```bash
# Start test containers
docker-compose -f docker-compose.test.yml up -d

# Check if containers are running
docker ps

# You should see:
# - smd-postgres-test
# - smd-backend-test
# - smd-nginx-test

# View logs
docker-compose -f docker-compose.test.yml logs -f
```

## Step 6: Run Multi-Catalog Migration

```bash
# Make script executable
chmod +x test-migration.sh

# Run migration
./test-migration.sh
```

Expected output:
```
✅ Docker is running
✅ Test containers are running
📦 Step 1: Generating Prisma Client...
✅ Prisma Client generated
🔄 Step 2: Running database migration...
✅ Migration completed
🌱 Step 3: Seeding default catalog...
✅ Default catalog seeded
```

## Step 7: Verify Migration

### 7.1 Check Database

```bash
# Access Prisma Studio
docker-compose -f docker-compose.test.yml exec backend-test npx prisma studio
```

Then open browser on your host machine:
```
http://192.168.1.100:5555
```

Verify:
- ✅ `catalogs` table exists with "Main Catalog"
- ✅ `catalog_customers` table has customer assignments
- ✅ All categories have `catalog_id` set
- ✅ All products have `catalog_id` set

### 7.2 Test API

From your host machine:

```bash
# Test health endpoint
curl http://192.168.1.100:3002/health

# Test login (use your test admin credentials)
curl -X POST http://192.168.1.100:3002/api/auth/login \
  -H "Content-Type: application/json" \
  -d '{"email":"admin@test.local","password":"TestAdmin123!"}'
```

## Step 8: Test Multi-Catalog Functionality

Once Phase 2 (Backend API) is complete, you'll test:

1. **Create New Catalog**
   ```bash
   curl -X POST http://192.168.1.100:3002/api/catalogs \
     -H "Authorization: Bearer <token>" \
     -H "Content-Type: application/json" \
     -d '{"name":"Test Catalog 2","slug":"test-2"}'
   ```

2. **Assign Customer to Catalog**
3. **Verify Catalog Isolation**
4. **Test Customer Login and Product Visibility**

## Step 9: Access from Host Machine

Update your host machine's `/etc/hosts` (optional):
```bash
sudo nano /etc/hosts
```

Add:
```
192.168.1.100  smd-test.local
```

Now you can access:
- Backend: `http://smd-test.local:3002`
- Nginx: `http://smd-test.local:8081`

## Step 10: Frontend Testing

Update frontend `.env` to point to test backend:

```bash
# In frontend directory
echo "VITE_API_URL=http://192.168.1.100:3002/api" > .env.local

# Start frontend
npm run dev
```

Access frontend at `http://localhost:5173` and it will connect to test backend.

## Troubleshooting

### Containers won't start
```bash
# Check logs
docker-compose -f docker-compose.test.yml logs

# Restart containers
docker-compose -f docker-compose.test.yml restart
```

### Migration fails
```bash
# Check database connection
docker-compose -f docker-compose.test.yml exec backend-test npx prisma db push

# View backend logs
docker-compose -f docker-compose.test.yml logs backend-test
```

### Can't access from host
```bash
# On VM, check if ports are listening
sudo netstat -tlnp | grep -E '3002|8081|5433'

# Check firewall
sudo ufw status
sudo ufw allow 3002
sudo ufw allow 8081
```

## Cleanup

When done testing:

```bash
# Stop containers
docker-compose -f docker-compose.test.yml down

# Remove volumes (WARNING: deletes all test data)
docker-compose -f docker-compose.test.yml down -v

# Remove images
docker-compose -f docker-compose.test.yml down --rmi all
```

## Production Deployment

After successful testing:

1. ✅ Verify all features work correctly
2. ✅ Document any issues or changes needed
3. ✅ Backup production database
4. ✅ Schedule maintenance window
5. ✅ Deploy to production using production scripts
6. ✅ Run production migration
7. ✅ Verify production deployment
8. ✅ Monitor for issues

## Port Reference

| Service | Production | Test | Description |
|---------|-----------|------|-------------|
| PostgreSQL | 5432 | 5433 | Database |
| Backend API | 3001 | 3002 | API Server |
| Nginx | 8080 | 8081 | Reverse Proxy |

## Environment Comparison

| Aspect | Production | Test |
|--------|-----------|------|
| Container Names | smd-* | smd-*-test |
| Network | smd-network | smd-test-network |
| Volumes | postgres_data | postgres_test_data |
| Database | smd_hardware | smd_hardware_test |
| Ports | Standard | +1 offset |

---

**Next Steps:** Once VM is ready, proceed with Phase 2 (Backend API implementation) and test each feature incrementally.
