import { Decimal } from '@prisma/client/runtime/library';

export interface FiscalItem {
    unitPrice: number | Decimal;
    quantity: number;
    tvaRate: number | Decimal;
    discount?: number | Decimal;
}

export interface FiscalTotals {
    totalHT: number;
    totalTVA: number;
    totalTTC: number;
    netMargin?: number;
}

/**
 * Calculates fiscal totals for a list of items.
 * Handles Decimal or number inputs safely.
 */
export const calculateFiscalTotals = (items: FiscalItem[], timbreFiscal: number = 1.000): FiscalTotals => {
    let totalHT = 0;
    let totalTVA = 0;

    items.forEach(item => {
        const price = Number(item.unitPrice);
        const qty = Number(item.quantity);
        const tva = Number(item.tvaRate);
        const discount = Number(item.discount || 0);

        // Calculate line HT (after discount)
        const lineTotalHT = (price * qty) - discount;

        // Calculate line TVA
        const lineTVA = lineTotalHT * (tva / 100);

        totalHT += lineTotalHT;
        totalTVA += lineTVA;
    });

    // Round to 3 decimal places (Tunisian standard)
    totalHT = Math.round(totalHT * 1000) / 1000;
    totalTVA = Math.round(totalTVA * 1000) / 1000;

    const totalTTC = totalHT + totalTVA + timbreFiscal;

    return {
        totalHT,
        totalTVA,
        totalTTC: Math.round(totalTTC * 1000) / 1000
    };
};

/**
 * Calculates margin for a single product sale
 */
export const calculateMargin = (sellingPriceHT: number, costPrice: number, quantity: number): number => {
    const margin = (sellingPriceHT - costPrice) * quantity;
    return Math.round(margin * 1000) / 1000;
};
