import prisma from '../config/database';
import { inventoryService } from '../services/inventory.service';
import {
    PrismaClient,
    Prisma,
    StockTransactionType
} from '@prisma/client';

// Enums are now imported top-level above

async function runVerification() {
    console.log('🚀 Starting Inventory Service Verification...');

    try {
        // 1. Setup: Get a product and a user
        let product = await prisma.product.findFirst();
        let user = await prisma.user.findFirst();

        if (!user) {
            console.log('⚠️ No users found. Creating dummy user...');
            user = await prisma.user.create({
                data: {
                    email: 'test@example.com',
                    passwordHash: 'dummy',
                    firstName: 'Test',
                    lastName: 'User'
                }
            });
        }

        if (!product) {
            console.log('⚠️ No products found. Creating dummy product...');
            // Need a category first
            let category = await prisma.category.findFirst();
            if (!category) {
                category = await prisma.category.create({
                    data: {
                        name: 'Test Category',
                        slug: 'test-category',
                    }
                });
            }

            product = await prisma.product.create({
                data: {
                    name: 'Test Product',
                    slug: 'test-product',
                    basePrice: 100,
                    categoryId: category.id,
                    sku: 'TEST-SKU-001'
                }
            });
        }

        console.log(`📝 Using Product: ${product.name} (${product.id})`);
        console.log(`👤 Using User: ${user.email} (${user.id})`);

        // 2. Test: Add Supplier Stock
        console.log('\n--- Test 1: Add Supplier Stock ---');
        const addedStock = await inventoryService.addSupplierStock(product.id, 50, 'Initial Test Stock');
        console.log('✅ Added 50 units. New Level:', addedStock.quantity);

        // 3. Test: Check Stock
        const currentStock = await inventoryService.checkSupplierStock(product.id);
        if (currentStock === addedStock.quantity) {
            console.log('✅ Check Stock matches:', currentStock);
        } else {
            console.error('❌ Stock mismatch!');
        }

        // 4. Test: Create Dummy Order (if possible/needed, or just simulate transfer params)
        // Since creating a full order graph is complex, we will test `deductSupplierStock` and `addToCustomerInventory` individually 
        // to simulate the key parts of `processOrderStockTransfer`.

        // Create a dummy order for sourceOrderId constraint
        const order = await prisma.order.create({
            data: {
                orderNumber: 'TEST-ORDER-001-' + Date.now(),
                customerName: 'Test User',
                customerEmail: user.email,
                customerPhone: '12345678',
                user: { connect: { id: user.id } },
                paymentMethod: 'CASH_ON_DELIVERY' as any, // Use string if Enum fails
                totalBrutHT: 100,
                totalNetHT: 100,
                totalTVA: 19,
                timbreFiscal: 0.600,
                netAPayer: 120.2,
                subtotal: 100,
                taxAmount: 19,
                totalAmount: 119.6
            }
        });

        console.log('\n--- Test 2: Deduct Stock (Simulate Sale) ---');
        const deductionQty = 5;
        const deducted = await inventoryService.deductSupplierStock(product.id, deductionQty, order.id);
        console.log(`✅ Deducted ${deductionQty}. New Level: `, deducted.quantity);

        console.log('\n--- Test 3: Add to Customer Inventory ---');
        const customerStock = await inventoryService.addToCustomerInventory(user.id, product.id, deductionQty, order.id);
        console.log(`✅ Customer received ${deductionQty}. Entry ID: `, customerStock.id);

        // 5. Test: Get Customer Inventory
        const custInv = await inventoryService.getCustomerInventory(user.id);
        console.log('✅ Customer Inventory Count:', custInv.length);
        console.log('   First Item Qty:', custInv[0].quantity);

        // 6. Test: Transaction Log
        const logs = await inventoryService.getStockTransactionHistory({ productId: product.id });
        console.log('\n--- Test 4: Audit Logs ---');
        console.log(`✅ Found ${logs.length} transactions for this product.`);
        logs.forEach(l => console.log(`   - [${l.type}] ${l.quantity} (Notes: ${l.notes})`));

        console.log('\n✨ Verification Complete!');

    } catch (error) {
        console.error('❌ Verification Failed:', error);
    } finally {
        await prisma.$disconnect();
    }
}

// Check if running directly
// In ESM we can check if the file url matches the executed script, or just call it if we are running the script.
// For simplicity in this env:
runVerification();

export default runVerification;
