
import { PrismaClient, OrderStatus } from '@prisma/client';
import { createOrder, updateOrderStatus } from '../controllers/order.controller.js';
import { createProduct, updateProductStock } from '../controllers/product.controller.js';

const prisma = new PrismaClient();

// Mock Express Request/Response
const mockResponse = () => {
    const res: any = {};
    res.status = (code: number) => {
        res.statusCode = code;
        return res;
    };
    res.json = (data: any) => {
        res.data = data;
        return res;
    };
    return res;
};

const mockNext = (err?: any) => {
    if (err) console.error('Next called with error:', err);
};

async function runIntegrationVerification() {
    console.log('🚀 Starting Integration Verification...');

    // 1. Setup User and Catalog
    const user = await prisma.user.create({
        data: {
            email: `test-integration-${Date.now()}@example.com`,
            passwordHash: 'hash',
            firstName: 'Integration',
            lastName: 'Tester',
            role: 'ADMIN' // Needed for creating products potentially?
        }
    });

    let catalog = await prisma.catalog.findFirst({ where: { isDefault: true } });
    if (!catalog) {
        console.log('⚠️ No default catalog found. Creating one...');
        catalog = await prisma.catalog.create({
            data: {
                name: 'Default Catalog',
                slug: 'default-catalog',
                isDefault: true,
                description: 'Created by Integration Test',
                defaultTvaRate: 19
            }
        });
    }

    // Mock Auth Request
    const authReq = (body: any = {}, params: any = {}, query: any = {}) => ({
        body,
        params,
        query,
        user: { id: user.id, role: user.role },
        headers: { 'x-catalog-id': catalog.id },
        header: (name: string) => {
            if (name.toLowerCase() === 'x-catalog-id') return catalog.id;
            return null;
        },
        get: (name: string) => { // Express alias for header
            if (name.toLowerCase() === 'x-catalog-id') return catalog.id;
            return null;
        }
    } as any);

    try {
        // ==========================================
        // TEST 1: Create Product with Initial Stock
        // ==========================================
        console.log('\n--- Test 1: Create Product with Stock ---');
        const productSlug = `integ-prod-${Date.now()}`;
        const productRes = mockResponse();

        await createProduct(authReq({
            name: 'Integration Product',
            slug: productSlug,
            sku: `SKU-${Date.now()}`,
            categoryId: (await prisma.category.findFirst())?.id,
            price: 100,
            stockQuantity: 10 // Initial Stock
        }), productRes, mockNext);

        if (productRes.statusCode === 201) {
            console.log('✅ Product Created:', productRes.data.data.id);
        } else {
            console.error('❌ Product Creation Failed:', productRes.data);
            throw new Error('Product creation failed');
        }

        const productId = productRes.data.data.id;

        // Verify Stock Transaction
        const audit1 = await prisma.stockTransaction.findFirst({
            where: { productId },
            orderBy: { createdAt: 'desc' }
        });
        console.log(`   Initial Audit Log: ${audit1 ? '✅ Found' : '❌ Missing'}`);
        if (audit1?.quantity !== 10) throw new Error('Audit log mismatch');

        // ==========================================
        // TEST 2: Create Order (Stock Validation)
        // ==========================================
        console.log('\n--- Test 2: Create Order (Valid Stock) ---');
        const orderRes = mockResponse();

        await createOrder(authReq({
            items: [{ productId, quantity: 5 }],
            shippingAddress: 'Test Address'
        }), orderRes);

        if (orderRes.statusCode === 201) {
            console.log('✅ Order Created:', orderRes.data.data.id);
        } else {
            console.error('❌ Order Creation Failed:', orderRes.data);
            throw new Error('Order creation failed');
        }
        const orderId = orderRes.data.data.id;

        // Verify NO stock deduction yet (just validation)
        const stockAfterOrder = await prisma.supplierInventory.findUnique({ where: { productId } });
        console.log(`   Stock after order creation: ${stockAfterOrder?.quantity} (Expected 10)`);
        if (stockAfterOrder?.quantity !== 10) throw new Error('Stock shouldn\'t be deducted yet');

        // ==========================================
        // TEST 3: Create Order (Insufficient Stock)
        // ==========================================
        console.log('\n--- Test 3: Create Order (Insufficient Stock) ---');
        const failOrderRes = mockResponse();
        await createOrder(authReq({
            items: [{ productId, quantity: 100 }] // Have 10, ask 100
        }), failOrderRes);

        if (failOrderRes.statusCode === 400 && failOrderRes.data.error.code === 'INSUFFICIENT_STOCK') {
            console.log('✅ Blocked Insufficient Stock Order');
        } else {
            console.error('❌ Failed to block insufficient stock:', failOrderRes.data || failOrderRes.statusCode);
            // Don't throw, just log
        }

        // ==========================================
        // TEST 4: Deliver Order (Stock Transfer)
        // ==========================================
        console.log('\n--- Test 4: Deliver Order -> Stock Transfer ---');
        const updateRes = mockResponse();
        await updateOrderStatus(authReq(
            { status: 'DELIVERED' }, // Body
            { id: orderId } // Params
        ), updateRes);

        if (updateRes.statusCode !== 200 && !updateRes.data?.success) {
            console.error('❌ Update Status Failed:', updateRes.data);
            throw new Error('Update status failed');
        }
        console.log('✅ Order Status Updated to DELIVERED');

        // Verify Stock Deduction
        const finalStock = await prisma.supplierInventory.findUnique({ where: { productId } });
        console.log(`   Final Supplier Stock: ${finalStock?.quantity} (Expected 5)`);
        if (finalStock?.quantity !== 5) throw new Error('Stock not deducted correctly');

        // Verify Customer Inventory
        const custInv = await prisma.customerInventory.findFirst({
            where: { customerId: user.id, productId }
        });
        console.log(`   Customer Inventory: ${custInv?.quantity} (Expected 5)`);
        if (custInv?.quantity !== 5) throw new Error('Customer inventory not updated');

        // ==========================================
        // TEST 5: Manual Stock Update (Admin)
        // ==========================================
        console.log('\n--- Test 5: Manual Stock Update ---');
        const stockUpdateRes = mockResponse();
        await updateProductStock(authReq(
            { stockQuantity: 20 },
            { id: productId }
        ), stockUpdateRes, mockNext);

        const adjustedStock = await prisma.supplierInventory.findUnique({ where: { productId } });
        console.log(`   Adjusted Stock: ${adjustedStock?.quantity} (Expected 20)`);
        if (adjustedStock?.quantity !== 20) throw new Error('Manual adjustment failed');

        // Verify Audit Log for adjustment
        const audit2 = await prisma.stockTransaction.findFirst({
            where: { productId, type: 'ADJUSTMENT' },
            orderBy: { createdAt: 'desc' }
        });
        console.log(`   Adjustment Audit Log: ${audit2 ? '✅ Found' : '❌ Missing'}`);


        console.log('\n✨ Integration Verification Complete!');

    } catch (error) {
        console.error('\n❌ Verification Process Failed:', error);
    } finally {
        // Cleanup
        //   await prisma.user.delete({ where: { id: user.id } });
        // Leaving data for manual inspection if needed, or stick to auto-cleanup in future
    }
}

// Run
runIntegrationVerification()
    .catch(console.error)
    .finally(() => prisma.$disconnect());
