import { Request, Response } from 'express';
import { PluginManager } from '../core/PluginManager.js';
// Note: In a real DI system we'd inject this. For now we might need to export the instance from server.ts
// OR we attach it to req.app.locals
// But typically for this codebase we can attach it to the Express app.

export const getPlugins = async (req: Request, res: Response) => {
    try {
        console.log('[DEBUG] getPlugins controller called');
        // Access plugin manager from app locals (we will set this in server.ts)
        const pluginManager = req.app.get('pluginManager') as PluginManager;
        console.log('[DEBUG] pluginManager retrieved:', !!pluginManager);

        if (!pluginManager) {
            console.error('[DEBUG] Plugin Manager missing');
            return res.status(500).json({ success: false, error: 'Plugin Manager not initialized' });
        }

        console.log('[DEBUG] Calling pluginManager.getPlugins()');
        // Check if getPlugins exists
        if (typeof pluginManager.getPlugins !== 'function') {
            console.error('[DEBUG] getPlugins is not a function on pluginManager:', pluginManager);
            throw new Error('pluginManager.getPlugins is not a function');
        }

        const rawPlugins = pluginManager.getPlugins();
        console.log('[DEBUG] rawPlugins:', rawPlugins ? rawPlugins.length : 'null');

        const plugins = rawPlugins.map((p: any) => ({
            name: p.name,
            version: p.version,
            status: p.status, // ACTIVE or INACTIVE
            description: p.description || 'System Plugin'
        }));

        res.json({
            success: true,
            data: plugins
        });
    } catch (error) {
        console.error('Get plugins error:', error);
        res.status(500).json({ success: false, error: 'Failed to fetch plugins' });
    }
};

export const togglePlugin = async (req: Request, res: Response) => {
    try {
        const { name } = req.params;
        // const { enabled } = req.body; // Later we can pass explicit state
        const pluginManager = req.app.get('pluginManager') as PluginManager;

        if (!pluginManager) {
            return res.status(500).json({ success: false, error: 'Plugin Manager not initialized' });
        }

        const plugin = pluginManager.getPlugins().find(p => p.name === name);
        if (!plugin) {
            return res.status(404).json({ success: false, error: 'Plugin not found' });
        }

        const newState = plugin.status !== 'ACTIVE'; // Toggle
        await pluginManager.togglePlugin(name, newState);

        res.json({
            success: true,
            data: {
                name,
                status: newState ? 'ACTIVE' : 'INACTIVE',
                message: 'Restart required to apply changes completely.'
            }
        });
    } catch (error) {
        console.error('Toggle plugin error:', error);
        res.status(500).json({ success: false, error: 'Failed to toggle plugin' });
    }
};

export const getNavigation = async (req: Request, res: Response) => {
    try {
        const pluginManager = req.app.get('pluginManager') as PluginManager;
        if (!pluginManager) return res.json({ success: true, data: [] });

        const nav = pluginManager.getNavigation();
        res.json({ success: true, data: nav });
    } catch (err) {
        res.status(500).json({ success: false, error: 'Failed to get nav' });
    }
};

export const getPluginConfig = async (req: Request, res: Response) => {
    try {
        const { name } = req.params;
        const pluginManager = req.app.get('pluginManager') as PluginManager;
        const config = await pluginManager.getPluginConfig(name);
        res.json({ success: true, data: config });
    } catch (err) {
        res.status(500).json({ success: false, error: 'Failed to get plugin config' });
    }
};

export const updatePluginConfig = async (req: Request, res: Response) => {
    try {
        const { name } = req.params;
        const config = req.body;
        const pluginManager = req.app.get('pluginManager') as PluginManager;

        await pluginManager.updatePluginConfig(name, config);
        res.json({ success: true, message: 'Configuration updated' });
    } catch (err) {
        res.status(500).json({ success: false, error: 'Failed to update plugin config' });
    }
};

export const uploadPlugin = async (req: Request, res: Response) => {
    try {
        if (!req.file) {
            return res.status(400).json({ success: false, error: 'No file uploaded' });
        }

        const pluginManager = req.app.get('pluginManager') as PluginManager;
        await pluginManager.installPlugin(req.file.path);

        res.json({
            success: true,
            message: 'Plugin uploaded and installed. Please restart the server to load it.'
        });
    } catch (err) {
        console.error('Upload Error:', err);
        res.status(500).json({ success: false, error: 'Failed to upload plugin' });
    }
};
