import { Request, Response } from 'express';
import { AuthRequest } from '../middleware/auth.js';
import { PrismaClient, OrderStatus, PaymentMethod, PaymentStatus } from '@prisma/client';
import { resolveCatalogContext } from '../utils/catalogContext.js';
import {
  calculateLineItemTva,
  calculateDocumentTva,
  getEffectiveTvaRate,
  isCustomerTvaExempt
} from '../utils/tvaCalculations.js';
import { InventoryService } from '../services/inventory.service.js';

const prisma = new PrismaClient();
const inventoryService = new InventoryService();

/**
 * Generate unique order number
 */
function generateOrderNumber(): string {
  const timestamp = Date.now().toString(36).toUpperCase();
  const random = Math.random().toString(36).substring(2, 6).toUpperCase();
  return `ORD-${timestamp}-${random}`;
}

/**
 * Create a new order
 * POST /api/orders
 */
export const createOrder = async (req: AuthRequest, res: Response) => {
  try {
    const {
      items,
      shippingAddressId,
      shippingAddress,
      paymentMethod,
      notes,
    } = req.body;

    // Resolve catalog context
    const catalog = await resolveCatalogContext(req, prisma);

    // Get authenticated user
    const userId = req.user?.id;
    if (!userId) {
      return res.status(401).json({
        success: false,
        error: {
          code: 'UNAUTHORIZED',
          message: 'User not authenticated',
        },
      });
    }

    // Fetch user details
    const user = await prisma.user.findUnique({
      where: { id: userId },
      include: {
        addresses: {
          where: { isDefault: true },
          take: 1,
        },
      },
    });

    if (!user) {
      return res.status(404).json({
        success: false,
        error: {
          code: 'USER_NOT_FOUND',
          message: 'User not found',
        },
      });
    }

    // Validate items
    if (!items || !Array.isArray(items) || items.length === 0) {
      return res.status(400).json({
        success: false,
        error: {
          code: 'INVALID_ITEMS',
          message: 'Order must contain at least one item',
        },
      });
    }

    // Fetch all products involved
    const productIds = items.map((i: any) => i.productId);
    const products = await prisma.product.findMany({
      where: { id: { in: productIds } }
    });

    // Map products for easy access
    const productMap = new Map(products.map(p => [p.id, p]));

    // Validate product existence and status, and prepare inventory check
    const checkItems = [];
    const orderItems = [];
    const lineItemResults = [];
    const customerTvaExempt = await isCustomerTvaExempt(userId);

    for (const item of items) {
      const product = productMap.get(item.productId);

      if (!product) {
        return res.status(404).json({
          success: false,
          error: {
            code: 'PRODUCT_NOT_FOUND',
            message: `Product ${item.productId} not found`,
          },
        });
      }

      // Verify product belongs to current catalog
      if (product.catalogId !== catalog.id) {
        return res.status(400).json({
          success: false,
          error: {
            code: 'INVALID_CATALOG_PRODUCT',
            message: `Product ${product.name} does not belong to the current catalog`,
          },
        });
      }

      if (!product.isActive) {
        return res.status(400).json({
          success: false,
          error: {
            code: 'PRODUCT_INACTIVE',
            message: `Product ${product.name} is not available`,
          },
        });
      }

      // Prepare for inventory check
      checkItems.push({
        productId: product.id,
        quantity: item.quantity,
        productName: product.name,
        // Add other mock properties if needed by OrderItem type, but InventoryService allows partial?
        // Wait, InventoryService signature says OrderItem[].
        // I should cast it or fix expected type. 
        // Ideally InventoryService should take { productId, quantity, productName? }[]
        // I will cast as any for now to satisfy TS if the runtime usage only needs these fields.
        id: 'temp',
        orderId: 'temp',
        tvaRate: product.tvaRate,
        tvaExempt: product.tvaExempt,
        productSku: product.sku,
        selectedSize: item.selectedSize || null,
        selectedUnitType: item.selectedUnitType || null,
        totalHT: new (await import('@prisma/client/runtime/library')).Decimal(0), // Mock
        tvaAmount: new (await import('@prisma/client/runtime/library')).Decimal(0),
        totalTTC: new (await import('@prisma/client/runtime/library')).Decimal(0),
        totalPrice: new (await import('@prisma/client/runtime/library')).Decimal(0),
        discount: new (await import('@prisma/client/runtime/library')).Decimal(0),
        createdAt: new Date(),
        updatedAt: new Date()
      });

      const unitPrice = Number(item.price || product.basePrice);

      // Get effective TVA rate
      const tvaInfo = await getEffectiveTvaRate(product.id, catalog.id);
      const effectiveTvaExempt = customerTvaExempt || tvaInfo.exempt;
      const effectiveTvaRate = effectiveTvaExempt ? 0 : tvaInfo.rate;

      // Calculate line item TVA
      const lineCalc = calculateLineItemTva({
        quantity: item.quantity,
        unitPrice: unitPrice,
        discount: 0,
        tvaRate: effectiveTvaRate,
        tvaExempt: effectiveTvaExempt,
        fodecSubject: product.fodecSubject,
      });

      lineItemResults.push(lineCalc);

      orderItems.push({
        productId: product.id,
        productName: product.name,
        productSku: product.sku,
        selectedSize: item.selectedSize || null,
        selectedUnitType: item.selectedUnitType || null,
        quantity: item.quantity,
        unitPrice,
        discount: lineCalc.discount,
        totalHT: lineCalc.totalNetHT,
        tvaRate: lineCalc.tvaRate,
        tvaAmount: lineCalc.tvaAmount,
        totalTTC: lineCalc.totalTTC,
        tvaExempt: lineCalc.tvaExempt,
        totalPrice: lineCalc.totalTTC,
      });
    }

    // [INVENTORY] Validate Stock Availability
    // We cast checkItems to any because we are providing necessary fields but constructing full OrderItem is tedious
    const stockValidation = await inventoryService.validateOrderStock(checkItems as any);

    if (!stockValidation.isValid) {
      return res.status(400).json({
        success: false,
        error: {
          code: 'INSUFFICIENT_STOCK',
          message: 'One or more items are out of stock',
          details: stockValidation.outOfStockItems
        }
      });
    }

    // Calculate document totals
    const tvaSummary = calculateDocumentTva(lineItemResults);

    // Add Timbre Fiscal
    const timbreFiscal = 1.000;
    const finalNetAPayer = tvaSummary.netAPayer + timbreFiscal;

    // Determine shipping address
    let finalShippingAddress = shippingAddress;
    let finalShippingAddressId = shippingAddressId;

    if (!finalShippingAddress && !finalShippingAddressId) {
      if (user.addresses.length > 0) {
        const defaultAddress = user.addresses[0];
        finalShippingAddressId = defaultAddress.id;
        finalShippingAddress = `${defaultAddress.street}, ${defaultAddress.city}${defaultAddress.postalCode ? ', ' + defaultAddress.postalCode : ''
          }, ${defaultAddress.country}`;
      }
    }

    // Create order with items
    const order = await prisma.order.create({
      data: {
        orderNumber: generateOrderNumber(),
        user: { connect: { id: user.id } },
        customerName: `${user.firstName || ''} ${user.lastName || ''}`.trim() || user.email,
        customerEmail: user.email,
        customerPhone: user.phone || '',
        shippingAddressId: finalShippingAddressId,
        shippingAddress: finalShippingAddress,
        status: OrderStatus.PENDING,
        paymentMethod: paymentMethod || PaymentMethod.CASH_ON_DELIVERY,
        paymentStatus: PaymentStatus.PENDING,

        // Fiscal Fields
        totalBrutHT: tvaSummary.totalBrutHT,
        totalDiscount: tvaSummary.totalDiscount,
        totalNetHT: tvaSummary.totalNetHT,
        tva19Amount: tvaSummary.tva19Amount,
        tva7Amount: tvaSummary.tva7Amount,
        tva0Amount: tvaSummary.tva0Amount,
        totalTVA: tvaSummary.totalTVA,
        netAPayer: finalNetAPayer,
        netAPayerWords: tvaSummary.netAPayerWords,
        customerTvaExempt,

        // Fiscal Snapshot
        fiscalTotalHT: tvaSummary.totalNetHT,
        fiscalTotalTVA: tvaSummary.totalTVA,
        timbreFiscal: timbreFiscal,
        matriculeFiscal: user.taxId,

        // Backward Compatibility
        subtotal: tvaSummary.totalNetHT,
        taxAmount: tvaSummary.totalTVA,
        totalAmount: finalNetAPayer,

        notes,
        items: {
          create: orderItems,
        },
        statusHistory: {
          create: {
            status: OrderStatus.PENDING,
            notes: 'Order created',
          },
        },
      },
      include: {
        items: {
          include: {
            product: {
              select: {
                id: true,
                name: true,
                slug: true,
                images: {
                  where: { isPrimary: true },
                  take: 1,
                },
              },
            },
          },
        },
        user: {
          select: {
            id: true,
            email: true,
            firstName: true,
            lastName: true,
            companyName: true,
          },
        },
      },
    });

    return res.status(201).json({
      success: true,
      data: order,
      message: 'Order created successfully',
    });
  } catch (error: any) {
    console.error('Create order error:', error);
    return res.status(500).json({
      success: false,
      error: {
        code: 'INTERNAL_ERROR',
        message: 'Failed to create order',
        details: error.message,
      },
    });
  }
};

/**
 * Get all orders (Admin only)
 * GET /api/orders
 */
export const getAllOrders = async (req: AuthRequest, res: Response) => {
  try {
    const {
      page = '1',
      limit = '20',
      status,
      search,
      userId,
    } = req.query;

    const pageNum = parseInt(page as string);
    const limitNum = parseInt(limit as string);
    const skip = (pageNum - 1) * limitNum;

    // Build where clause
    const catalog = await resolveCatalogContext(req, prisma);
    const where: any = {
      items: {
        some: {
          product: {
            catalogId: catalog.id,
          },
        },
      },
    };

    if (status) {
      where.status = status;
    }

    if (userId) {
      where.userId = userId;
    }

    if (search) {
      where.OR = [
        { orderNumber: { contains: search as string, mode: 'insensitive' } },
        { customerName: { contains: search as string, mode: 'insensitive' } },
        { customerEmail: { contains: search as string, mode: 'insensitive' } },
        { customerPhone: { contains: search as string, mode: 'insensitive' } },
      ];
    }

    // Get orders with pagination
    const [orders, total] = await Promise.all([
      prisma.order.findMany({
        where,
        include: {
          items: {
            include: {
              product: {
                select: {
                  id: true,
                  name: true,
                  images: {
                    where: { isPrimary: true },
                    take: 1,
                  },
                },
              },
            },
          },
          user: {
            select: {
              id: true,
              email: true,
              firstName: true,
              lastName: true,
              companyName: true,
              customerType: true,
            },
          },
          _count: {
            select: {
              items: true,
            },
          },
        },
        orderBy: {
          createdAt: 'desc',
        },
        skip,
        take: limitNum,
      }),
      prisma.order.count({ where }),
    ]);

    return res.json({
      success: true,
      data: {
        orders,
        pagination: {
          page: pageNum,
          limit: limitNum,
          total,
          totalPages: Math.ceil(total / limitNum),
        },
      },
    });
  } catch (error: any) {
    console.error('Get orders error:', error);
    return res.status(500).json({
      success: false,
      error: {
        code: 'INTERNAL_ERROR',
        message: 'Failed to fetch orders',
        details: error.message,
      },
    });
  }
};

/**
 * Get order by ID
 * GET /api/orders/:id
 */
export const getOrderById = async (req: AuthRequest, res: Response) => {
  try {
    const { id } = req.params;
    const userId = req.user?.id;
    const userRole = req.user?.role;

    const order = await prisma.order.findUnique({
      where: { id },
      include: {
        items: {
          include: {
            product: {
              select: {
                id: true,
                name: true,
                slug: true,
                sku: true,
                images: {
                  where: { isPrimary: true },
                  take: 1,
                },
              },
            },
          },
        },
        user: {
          select: {
            id: true,
            email: true,
            firstName: true,
            lastName: true,
            phone: true,
            companyName: true,
            customerType: true,
          },
        },
        statusHistory: {
          orderBy: {
            createdAt: 'desc',
          },
        },
      },
    });

    if (!order) {
      return res.status(404).json({
        success: false,
        error: {
          code: 'ORDER_NOT_FOUND',
          message: 'Order not found',
        },
      });
    }

    // Check authorization (customer can only see their own orders)
    if (userRole === 'CUSTOMER' && order.userId !== userId) {
      return res.status(403).json({
        success: false,
        error: {
          code: 'FORBIDDEN',
          message: 'You do not have permission to view this order',
        },
      });
    }

    return res.json({
      success: true,
      data: order,
    });
  } catch (error: any) {
    console.error('Get order error:', error);
    return res.status(500).json({
      success: false,
      error: {
        code: 'INTERNAL_ERROR',
        message: 'Failed to fetch order',
        details: error.message,
      },
    });
  }
};

/**
 * Get customer's orders
 * GET /api/orders/my-orders
 */
export const getMyOrders = async (req: AuthRequest, res: Response) => {
  try {
    const userId = req.user?.id;

    if (!userId) {
      return res.status(401).json({
        success: false,
        error: {
          code: 'UNAUTHORIZED',
          message: 'User not authenticated',
        },
      });
    }

    const {
      page = '1',
      limit = '20',
      status,
    } = req.query;

    const catalog = await resolveCatalogContext(req, prisma);

    const pageNum = parseInt(page as string);
    const limitNum = parseInt(limit as string);
    const skip = (pageNum - 1) * limitNum;

    const where: any = {
      userId,
      items: {
        some: {
          product: {
            catalogId: catalog.id
          }
        }
      }
    };

    if (status) {
      where.status = status;
    }

    const [orders, total] = await Promise.all([
      prisma.order.findMany({
        where,
        include: {
          items: {
            include: {
              product: {
                select: {
                  id: true,
                  name: true,
                  slug: true,
                  images: {
                    where: { isPrimary: true },
                    take: 1,
                  },
                },
              },
            },
          },
          _count: {
            select: {
              items: true,
            },
          },
        },
        orderBy: {
          createdAt: 'desc',
        },
        skip,
        take: limitNum,
      }),
      prisma.order.count({ where }),
    ]);

    return res.json({
      success: true,
      data: {
        orders,
        pagination: {
          page: pageNum,
          limit: limitNum,
          total,
          totalPages: Math.ceil(total / limitNum),
        },
      },
    });
  } catch (error: any) {
    console.error('Get my orders error:', error);
    return res.status(500).json({
      success: false,
      error: {
        code: 'INTERNAL_ERROR',
        message: 'Failed to fetch orders',
        details: error.message,
      },
    });
  }
};

/**
 * Update order status (Admin only)
 * PUT /api/orders/:id/status
 */
export const updateOrderStatus = async (req: AuthRequest, res: Response) => {
  try {
    const { id } = req.params;
    const { status, notes } = req.body;

    if (!status || !Object.values(OrderStatus).includes(status)) {
      return res.status(400).json({
        success: false,
        error: {
          code: 'INVALID_STATUS',
          message: 'Invalid order status',
        },
      });
    }

    const existingOrder = await prisma.order.findUnique({ where: { id } });

    if (!existingOrder) {
      return res.status(404).json({
        success: false,
        error: {
          code: 'ORDER_NOT_FOUND',
          message: 'Order not found',
        },
      });
    }

    if (status === OrderStatus.DELIVERED && existingOrder.status !== OrderStatus.DELIVERED) {
      // [INVENTORY] Process Stock Transfer (Supplier -> Customer)
      try {
        await inventoryService.processOrderStockTransfer(id);
      } catch (err: any) {
        console.error('Inventory transfer failed:', err);
        return res.status(400).json({
          success: false,
          error: {
            code: 'INVENTORY_ERROR',
            message: 'Failed to transfer stock. Insufficient inventory.',
            details: err.message
          }
        });
      }
    }

    const order = await prisma.order.update({
      where: { id },
      data: {
        status,
        statusHistory: {
          create: {
            status,
            notes: notes || `Status updated to ${status}`,
          },
        },
      },
      include: {
        items: {
          include: {
            product: true,
          },
        },
        user: {
          select: {
            id: true,
            email: true,
            firstName: true,
            lastName: true,
            companyName: true,
          },
        },
        statusHistory: {
          orderBy: {
            createdAt: 'desc',
          },
        },
      },
    });

    return res.json({
      success: true,
      data: order,
      message: 'Order status updated successfully',
    });
  } catch (error: any) {
    console.error('Update order status error:', error);
    return res.status(500).json({
      success: false,
      error: {
        code: 'INTERNAL_ERROR',
        message: 'Failed to update order status',
        details: error.message,
      },
    });
  }
};

/**
 * Cancel order
 * PUT /api/orders/:id/cancel
 */
export const cancelOrder = async (req: AuthRequest, res: Response) => {
  try {
    const { id } = req.params;
    const { reason } = req.body;
    const userId = req.user?.id;
    const userRole = req.user?.role;

    const order = await prisma.order.findUnique({
      where: { id },
    });

    if (!order) {
      return res.status(404).json({
        success: false,
        error: {
          code: 'ORDER_NOT_FOUND',
          message: 'Order not found',
        },
      });
    }

    // Check authorization
    if (userRole === 'CUSTOMER' && order.userId !== userId) {
      return res.status(403).json({
        success: false,
        error: {
          code: 'FORBIDDEN',
          message: 'You do not have permission to cancel this order',
        },
      });
    }

    // Check if order can be cancelled
    if (order.status === OrderStatus.DELIVERED || order.status === OrderStatus.CANCELLED) {
      return res.status(400).json({
        success: false,
        error: {
          code: 'CANNOT_CANCEL',
          message: `Cannot cancel order with status ${order.status}`,
        },
      });
    }

    const updatedOrder = await prisma.order.update({
      where: { id },
      data: {
        status: OrderStatus.CANCELLED,
        statusHistory: {
          create: {
            status: OrderStatus.CANCELLED,
            notes: reason || 'Order cancelled by user',
          },
        },
      },
      include: {
        items: {
          include: {
            product: true,
          },
        },
        user: {
          select: {
            id: true,
            email: true,
            firstName: true,
            lastName: true,
          },
        },
      },
    });

    return res.json({
      success: true,
      data: updatedOrder,
      message: 'Order cancelled successfully',
    });
  } catch (error: any) {
    console.error('Cancel order error:', error);
    return res.status(500).json({
      success: false,
      error: {
        code: 'INTERNAL_ERROR',
        message: 'Failed to cancel order',
        details: error.message,
      },
    });
  }
};

export const updatePaymentStatus = async (req: AuthRequest, res: Response) => {
  try {
    const { id } = req.params;
    const { status, notes } = req.body;

    if (!status) {
      return res.status(400).json({ success: false, error: { message: 'Status is required' } });
    }

    const order = await prisma.order.findUnique({ where: { id } });
    if (!order) {
      return res.status(404).json({ success: false, error: { message: 'Order not found' } });
    }

    const updatedOrder = await prisma.order.update({
      where: { id },
      data: {
        paymentStatus: status,
        statusHistory: {
          create: {
            status: order.status, // Keep same order status
            notes: `Payment status updated to ${status}. ${notes || ''}`,
          }
        }
      },
      include: {
        items: true,
        user: true,
        statusHistory: {
          orderBy: { createdAt: 'desc' },
        },
      },
    });

    res.json({ success: true, data: updatedOrder });
  } catch (error) {
    console.error('Error updating payment status:', error);
    res.status(500).json({ success: false, error: { message: 'Failed to update payment status' } });
  }
};
