#!/bin/bash

# Multi-Catalog Migration Script for Docker Environment
# This script runs the Prisma migration and seeds the default catalog

set -e  # Exit on error

echo "========================================="
echo "Multi-Catalog Migration Script"
echo "========================================="
echo ""

# Check if Docker is running
if ! docker ps > /dev/null 2>&1; then
    echo "❌ Error: Docker is not running or you don't have permission to access it."
    echo "Please start Docker and try again."
    exit 1
fi

# Check if backend container exists
if ! docker ps -a | grep -q smd-backend; then
    echo "❌ Error: smd-backend container not found."
    echo "Please start your Docker containers first:"
    echo "  cd backend && docker-compose up -d"
    exit 1
fi

echo "✅ Docker is running"
echo ""

# Step 1: Generate Prisma Client
echo "📦 Step 1: Generating Prisma Client..."
docker-compose exec -T backend npx prisma generate
echo "✅ Prisma Client generated"
echo ""

# Step 2: Run migration
echo "🔄 Step 2: Running database migration..."
docker-compose exec -T backend npx prisma migrate deploy
echo "✅ Migration completed"
echo ""

# Step 3: Seed default catalog
echo "🌱 Step 3: Seeding default catalog..."
docker-compose exec -T backend npx ts-node prisma/seed-default-catalog.ts
echo "✅ Default catalog seeded"
echo ""

echo "========================================="
echo "✅ Multi-Catalog Migration Complete!"
echo "========================================="
echo ""
echo "Next steps:"
echo "1. Restart your backend container:"
echo "   docker-compose restart backend"
echo ""
echo "2. Verify the migration:"
echo "   docker-compose exec backend npx prisma studio"
echo ""
