import { describe, it, expect } from '@jest/globals';
import {
    calculateLineItemTva,
    calculateDocumentTva,
    numberToFrenchWords,
    TVA_RATES,
} from '../src/utils/tvaCalculations';

describe('TVA Calculation Engine', () => {
    describe('calculateLineItemTva', () => {
        it('should calculate TVA correctly for standard 19% rate', () => {
            const result = calculateLineItemTva({
                quantity: 10,
                unitPrice: 100,
                discount: 0,
                tvaRate: 19,
                tvaExempt: false,
            });

            expect(result.totalBrutHT).toBe(1000);
            expect(result.totalNetHT).toBe(1000);
            expect(result.tvaAmount).toBe(190);
            expect(result.totalTTC).toBe(1190);
            expect(result.tvaExempt).toBe(false);
        });

        it('should calculate TVA correctly for reduced 7% rate', () => {
            const result = calculateLineItemTva({
                quantity: 5,
                unitPrice: 50,
                discount: 0,
                tvaRate: 7,
                tvaExempt: false,
            });

            expect(result.totalBrutHT).toBe(250);
            expect(result.totalNetHT).toBe(250);
            expect(result.tvaAmount).toBe(17.5);
            expect(result.totalTTC).toBe(267.5);
        });

        it('should handle exempt products (0% TVA)', () => {
            const result = calculateLineItemTva({
                quantity: 10,
                unitPrice: 100,
                discount: 0,
                tvaRate: 0,
                tvaExempt: true,
            });

            expect(result.totalBrutHT).toBe(1000);
            expect(result.totalNetHT).toBe(1000);
            expect(result.tvaAmount).toBe(0);
            expect(result.totalTTC).toBe(1000);
            expect(result.tvaExempt).toBe(true);
        });

        it('should apply discounts correctly', () => {
            const result = calculateLineItemTva({
                quantity: 10,
                unitPrice: 100,
                discount: 100,
                tvaRate: 19,
                tvaExempt: false,
            });

            expect(result.totalBrutHT).toBe(1000);
            expect(result.discount).toBe(100);
            expect(result.totalNetHT).toBe(900);
            expect(result.tvaAmount).toBe(171); // 900 * 0.19
            expect(result.totalTTC).toBe(1071);
        });

        it('should handle decimal quantities and prices', () => {
            const result = calculateLineItemTva({
                quantity: 2.5,
                unitPrice: 45.99,
                discount: 0,
                tvaRate: 19,
                tvaExempt: false,
            });

            expect(result.totalBrutHT).toBeCloseTo(114.975, 2);
            expect(result.tvaAmount).toBeCloseTo(21.845, 2);
            expect(result.totalTTC).toBeCloseTo(136.82, 2);
        });
    });

    describe('calculateDocumentTva', () => {
        it('should aggregate multiple line items correctly', () => {
            const lineItems = [
                calculateLineItemTva({
                    quantity: 10,
                    unitPrice: 100,
                    discount: 0,
                    tvaRate: 19,
                    tvaExempt: false,
                }),
                calculateLineItemTva({
                    quantity: 5,
                    unitPrice: 50,
                    discount: 0,
                    tvaRate: 7,
                    tvaExempt: false,
                }),
                calculateLineItemTva({
                    quantity: 3,
                    unitPrice: 30,
                    discount: 0,
                    tvaRate: 0,
                    tvaExempt: true,
                }),
            ];

            const result = calculateDocumentTva(lineItems);

            expect(result.totalBrutHT).toBe(1340); // 1000 + 250 + 90
            expect(result.totalNetHT).toBe(1340);
            expect(result.tva19Amount).toBe(190); // 1000 * 0.19
            expect(result.tva7Amount).toBe(17.5); // 250 * 0.07
            expect(result.tva0Amount).toBe(0);
            expect(result.totalTVA).toBe(207.5);
            expect(result.netAPayer).toBe(1547.5);
        });

        it('should categorize TVA by rate correctly', () => {
            const lineItems = [
                calculateLineItemTva({
                    quantity: 1,
                    unitPrice: 100,
                    discount: 0,
                    tvaRate: 19,
                    tvaExempt: false,
                }),
                calculateLineItemTva({
                    quantity: 1,
                    unitPrice: 100,
                    discount: 0,
                    tvaRate: 19,
                    tvaExempt: false,
                }),
                calculateLineItemTva({
                    quantity: 1,
                    unitPrice: 100,
                    discount: 0,
                    tvaRate: 7,
                    tvaExempt: false,
                }),
            ];

            const result = calculateDocumentTva(lineItems);

            expect(result.tva19Amount).toBe(38); // 200 * 0.19
            expect(result.tva7Amount).toBe(7); // 100 * 0.07
            expect(result.totalTVA).toBe(45);
        });

        it('should handle all exempt items', () => {
            const lineItems = [
                calculateLineItemTva({
                    quantity: 10,
                    unitPrice: 50,
                    discount: 0,
                    tvaRate: 0,
                    tvaExempt: true,
                }),
            ];

            const result = calculateDocumentTva(lineItems);

            expect(result.totalNetHT).toBe(500);
            expect(result.totalTVA).toBe(0);
            expect(result.netAPayer).toBe(500);
        });
    });

    describe('numberToFrenchWords', () => {
        it('should convert simple numbers correctly', () => {
            expect(numberToFrenchWords(0)).toBe('Zéro dinar');
            expect(numberToFrenchWords(1)).toBe('Un dinar');
            expect(numberToFrenchWords(10)).toBe('Dix dinars');
            expect(numberToFrenchWords(100)).toBe('Cent dinars');
        });

        it('should handle numbers with millimes', () => {
            const result = numberToFrenchWords(123.456);
            expect(result).toContain('cent vingt-trois dinars');
            expect(result).toContain('millimes');
        });

        it('should handle thousands', () => {
            const result = numberToFrenchWords(1000);
            expect(result).toContain('mille');
        });

        it('should handle complex amounts', () => {
            const result = numberToFrenchWords(1234.56);
            expect(result).toContain('mille');
            expect(result).toContain('deux cent');
            expect(result).toContain('trente');
            expect(result).toContain('dinars');
        });

        it('should handle zero millimes', () => {
            const result = numberToFrenchWords(100.0);
            expect(result).toBe('Cent dinars');
        });
    });

    describe('TVA_RATES constants', () => {
        it('should have correct rate values', () => {
            expect(TVA_RATES.STANDARD).toBe(19);
            expect(TVA_RATES.REDUCED).toBe(7);
            expect(TVA_RATES.EXEMPT).toBe(0);
        });
    });

    describe('Edge Cases', () => {
        it('should handle zero quantity', () => {
            const result = calculateLineItemTva({
                quantity: 0,
                unitPrice: 100,
                discount: 0,
                tvaRate: 19,
                tvaExempt: false,
            });

            expect(result.totalBrutHT).toBe(0);
            expect(result.tvaAmount).toBe(0);
            expect(result.totalTTC).toBe(0);
        });

        it('should handle zero price', () => {
            const result = calculateLineItemTva({
                quantity: 10,
                unitPrice: 0,
                discount: 0,
                tvaRate: 19,
                tvaExempt: false,
            });

            expect(result.totalBrutHT).toBe(0);
            expect(result.tvaAmount).toBe(0);
            expect(result.totalTTC).toBe(0);
        });

        it('should handle discount equal to total', () => {
            const result = calculateLineItemTva({
                quantity: 10,
                unitPrice: 100,
                discount: 1000,
                tvaRate: 19,
                tvaExempt: false,
            });

            expect(result.totalNetHT).toBe(0);
            expect(result.tvaAmount).toBe(0);
            expect(result.totalTTC).toBe(0);
        });

        it('should handle very large numbers', () => {
            const result = calculateLineItemTva({
                quantity: 1000,
                unitPrice: 10000,
                discount: 0,
                tvaRate: 19,
                tvaExempt: false,
            });

            expect(result.totalBrutHT).toBe(10000000);
            expect(result.tvaAmount).toBe(1900000);
            expect(result.totalTTC).toBe(11900000);
        });
    });

    describe('Real-world Scenarios', () => {
        it('should calculate quotation with mixed products correctly', () => {
            // Scenario: Hardware store quotation
            // - 10 screws at 2 TND each (19% TVA)
            // - 5 medical masks at 10 TND each (7% TVA)
            // - 2 agricultural products at 50 TND each (0% TVA - exempt)

            const screws = calculateLineItemTva({
                quantity: 10,
                unitPrice: 2,
                discount: 0,
                tvaRate: 19,
                tvaExempt: false,
            });

            const masks = calculateLineItemTva({
                quantity: 5,
                unitPrice: 10,
                discount: 0,
                tvaRate: 7,
                tvaExempt: false,
            });

            const agricultural = calculateLineItemTva({
                quantity: 2,
                unitPrice: 50,
                discount: 0,
                tvaRate: 0,
                tvaExempt: true,
            });

            const quotation = calculateDocumentTva([screws, masks, agricultural]);

            expect(quotation.totalBrutHT).toBe(170); // 20 + 50 + 100
            expect(quotation.tva19Amount).toBe(3.8); // 20 * 0.19
            expect(quotation.tva7Amount).toBe(3.5); // 50 * 0.07
            expect(quotation.tva0Amount).toBe(0);
            expect(quotation.totalTVA).toBe(7.3);
            expect(quotation.netAPayer).toBe(177.3);
        });

        it('should handle customer with discount', () => {
            // Scenario: Loyal customer gets 10% discount on order
            const item = calculateLineItemTva({
                quantity: 10,
                unitPrice: 100,
                discount: 100, // 10% of 1000
                tvaRate: 19,
                tvaExempt: false,
            });

            expect(item.totalBrutHT).toBe(1000);
            expect(item.discount).toBe(100);
            expect(item.totalNetHT).toBe(900);
            expect(item.tvaAmount).toBe(171); // 900 * 0.19
            expect(item.totalTTC).toBe(1071);
        });
    });
});
