import { Express } from 'express';
import { HookSystem } from '../../core/HookSystem.js';
import * as logic from './logic.js';

export const name = 'tax-tunisia';
export const version = '1.0.0';
export const description = 'Tunisian Tax System (19%, 7%, FODEC)';

export const navigation = [
    {
        name: 'Tax Configuration',
        href: '/settings/taxes', // We will reuse settings for now or create new
        icon: 'Calculator', // String name of icon
        roles: ['SUPER_ADMIN', 'ADMIN', 'ACCOUNTING_ADMIN']
    }
];

export const init = async (app: Express, hooks: HookSystem) => {

    // Helper to get current config
    const getConfig = async () => {
        const pm = (app.get('pluginManager') as any);
        const config = await pm.getPluginConfig('tax-tunisia');
        // Merge with defaults
        return {
            standardRate: 19,
            reducedRate: 7,
            fodecRate: 1,
            enableStamp: true,
            stampValue: 1.000,
            ...config
        };
    };

    // Hook: calculate.line_item
    // Context: LineItemInput & { result?: LineItemResult }
    hooks.on('calculate.line_item', async (context: any, next) => {
        try {
            const config = await getConfig();
            // Apply Tunisian Tax Logic with Config
            const result = logic.calculateLineItem(context, config);
            context.result = result;
        } catch (error) {
            console.error('[TaxTunisia] Error in calculate.line_item:', error);
            throw error;
        }
        await next();
    });

    // Hook: calculate.document
    // Context: { lineItems: LineItemResult[], result?: DocumentTvaSummary }
    hooks.on('calculate.document', async (context: any, next) => {
        try {
            const config = await getConfig();
            const result = logic.calculateDocumentTva(context.lineItems, config);
            context.result = result;
        } catch (error) {
            console.error('[TaxTunisia] Error in calculate.document:', error);
            throw error;
        }
        await next();
    });

    // Hook: tva.get_effective_rate
    // Context: { productId: string, catalogId: string, result?: { rate: number, ... } }
    hooks.on('tva.get_effective_rate', async (context: any, next) => {
        try {
            const result = await logic.getEffectiveTvaRate(context.productId, context.catalogId);
            context.result = result;
        } catch (error) {
            console.error('[TaxTunisia] Error in tva.get_effective_rate:', error);
        }
        await next();
    });

    // Hook: tva.is_exempt
    // Context: { userId: string, isExempt?: boolean }
    hooks.on('tva.is_exempt', async (context: any, next) => {
        try {
            context.isExempt = await logic.isCustomerTvaExempt(context.userId);
        } catch (error) {
            console.error('[TaxTunisia] Error in tva.is_exempt:', error);
        }
        await next();
    });

    console.log('[TaxTunisia] Plugin initialized');
};
