import { createContext, useContext, useState, useEffect, ReactNode } from 'react';
import { useAuth } from './AuthContext';
import { buildAuthHeaders } from '../utils/apiClient';

type Language = 'en' | 'fr';

interface LanguageContextType {
  language: Language;
  setLanguage: (lang: Language) => void;
  t: any;
}

const LanguageContext = createContext<LanguageContextType | undefined>(undefined);

export const translations = {
  en: {
    // Sidebar Navigation
    nav: {
      dashboard: 'Dashboard',
      products: 'Products',
      categories: 'Categories',
      catalogs: 'Catalogs',
      orders: 'Orders',
      quotations: 'Quotations',
      customers: 'Customers',
      marketing: 'Marketing',
      accounting: 'Accounting',
      settings: 'Settings',
      logout: 'Logout',
    },
    // Dashboard Page
    dashboard: {
      title: 'Dashboard',
      welcome: 'Welcome back',
      overview: 'Overview',
      totalRevenue: 'Total Revenue',
      totalOrders: 'Total Orders',
      totalCustomers: 'Total Customers',
      pendingQuotations: 'Pending Quotations',
      recentOrders: 'Recent Orders',
      recentQuotations: 'Recent Quotations',
      viewAll: 'View All',
      orderNumber: 'Order',
      customer: 'Customer',
      amount: 'Amount',
      status: 'Status',
      date: 'Date',
      actions: 'Actions',
      viewDetails: 'View Details',
      noOrders: 'No orders yet',
      noQuotations: 'No quotations yet',
    },
    // Products Page
    products: {
      title: 'Products',
      addProduct: 'Add Product',
      searchPlaceholder: 'Search products...',
      name: 'Name',
      category: 'Category',
      price: 'Price',
      stock: 'Stock',
      status: 'Status',
      actions: 'Actions',
      edit: 'Edit',
      delete: 'Delete',
      active: 'Active',
      inactive: 'Inactive',
      inStock: 'In Stock',
      outOfStock: 'Out of Stock',
      lowStock: 'Low Stock',
      // Product Creation/Edit Form
      createNew: 'Create New Product',
      editProduct: 'Edit Product',
      addNewProduct: 'Add a new product to your catalog',
      updateProductInfo: 'Update product information',
      preview: 'Preview',
      saveAsDraft: 'Save as Draft',
      createProduct: 'Create Product',
      updateProduct: 'Update Product',
      saving: 'Saving...',
      updating: 'Updating...',
      // Tabs
      productInfo: 'Product Info',
      measurementSelling: 'Measurement & Selling',
      sizesSpecs: 'Sizes & Specs',
      pricingInventory: 'Pricing & Inventory',
      seo: 'SEO',
      // Product Info Section
      productName: 'Product Name',
      productNamePlaceholder: 'e.g., Cordless Drill 18V',
      selectCategory: 'Select a category',
      description: 'Description',
      descriptionPlaceholder: 'Detailed product description...',
      characters: 'characters',
      productImages: 'Product Images',
      uploadImage: 'Upload image',
      uploadFile: 'Upload File',
      useUrl: 'Use URL',
      firstImagePrimary: 'First image will be the primary image. Recommended size: 800x800px',
      brand: 'Brand',
      brandPlaceholder: 'e.g., Bosch, DeWalt, Makita',
      productStatus: 'Product Status',
      draft: 'Draft',
      hidden: 'Hidden',
      // Measurement Section
      sellingType: 'Selling Type',
      piece: 'Piece',
      weight: 'Weight',
      length: 'Length',
      volume: 'Volume',
      custom: 'Custom',
      customUnit: 'Custom Unit',
      customUnitPlaceholder: 'e.g., box, roll, sheet',
      packSizes: 'Pack Sizes',
      packSizesDesc: 'Define different pack quantities for this product',
      addPackSize: 'Add Pack Size',
      quantity: 'Quantity',
      label: 'Label',
      // Size Specifications
      hasSizes: 'This product has multiple sizes',
      sizeVariations: 'Size Variations',
      sizeVariationsDesc: 'Define different sizes and their specifications',
      addSize: 'Add Size',
      sizeName: 'Size Name',
      dimension: 'Dimension',
      // Pricing & Inventory
      basePrice: 'Base Price',
      sku: 'SKU',
      skuPlaceholder: 'e.g., DRL-18V-001',
      stockQuantity: 'Stock Quantity',
      lowStockThreshold: 'Low Stock Threshold',
      lowStockThresholdDesc: 'Get notified when stock falls below this number',
      // SEO Section
      metaTitle: 'Meta Title',
      metaTitlePlaceholder: 'SEO optimized title',
      metaDescription: 'Meta Description',
      metaDescriptionPlaceholder: 'SEO optimized description',
      slug: 'URL Slug',
      slugPlaceholder: 'product-url-slug',
      // Navigation
      previous: 'Previous',
      next: 'Next',
      // Validation
      productNameRequired: 'Product name is required',
      categoryRequired: 'Category is required',
      basePriceRequired: 'Base price must be greater than 0',
      skuRequired: 'SKU is required',
      fillRequiredFields: 'Please fill in all required fields',
      selectCategoryFirst: 'Please select a category',
      enterValidPrice: 'Please enter a valid price',
      // Success Messages
      productCreated: 'Product created successfully!',
      productUpdated: 'Product updated successfully!',
      productSavedDraft: 'Product saved as draft successfully!',
      // Error Messages
      failedToCreate: 'Failed to create product',
      failedToUpdate: 'Failed to update product',
      failedToLoad: 'Failed to load product',
    },
    // Categories Page
    categories: {
      title: 'Categories',
      addCategory: 'Add Category',
      name: 'Name',
      products: 'Products',
      status: 'Status',
      actions: 'Actions',
      edit: 'Edit',
      delete: 'Delete',
    },
    // Orders Page
    orders: {
      title: 'Orders',
      orderNumber: 'Order Number',
      customer: 'Customer',
      date: 'Date',
      total: 'Total',
      status: 'Status',
      actions: 'Actions',
      viewDetails: 'View Details',
      filterByStatus: 'Filter by Status',
      allStatuses: 'All Statuses',
      pending: 'Pending',
      processing: 'Processing',
      shipped: 'Shipped',
      delivered: 'Delivered',
      cancelled: 'Cancelled',
      editOrder: 'Edit Order',
      modifyItems: 'Modify items before changing status',
      addProduct: 'Add Product',
      searchProducts: 'Search products by name or SKU...',
      orderItems: 'Order Items',
      changeFromOriginal: 'Change from original',
      saveChanges: 'Save Changes',
    },
    // Quotations Page
    quotations: {
      title: 'Quotations',
      quotationNumber: 'Quotation Number',
      customer: 'Customer',
      date: 'Date',
      total: 'Total',
      status: 'Status',
      actions: 'Actions',
      viewDetails: 'View Details',
      approve: 'Approve',
      decline: 'Decline',
      pendingApproval: 'Pending Approval',
      approved: 'Approved',
      declined: 'Declined',
      filterByStatus: 'Filter by Status',
      allStatuses: 'All Statuses',
      convertedToOrder: 'Converted to Order',
      quotationWillExceed: 'This quotation will exceed the customer\'s credit limit by',
      creditLimit: 'Credit Limit',
      currentOutstanding: 'Current Outstanding',
      newOutstanding: 'New Outstanding',
      overLimitBy: 'Over Limit By',
      customerInformation: 'Customer Information',
      accountStatus: 'Account Status',
      items: 'Items',
      sku: 'SKU',
      quantity: 'Quantity',
      unitPrice: 'Unit Price',
      financialSummary: 'Financial Summary',
      subtotal: 'Subtotal',
      tax: 'Tax',
      created: 'Created',
      availableAfter: 'Available After',
      creditLimitExceeded: 'This quotation will exceed the customer\'s credit limit by {amount}',
      withinCreditLimit: 'Within credit limit. Available after: {amount}',
      confirmApprove: 'Are you sure you want to approve this quotation?',
      approveFailed: 'Failed to approve quotation',
      declineReasonRequired: 'Please provide a reason for declining',
      declineFailed: 'Failed to decline quotation',
    },
    // Customers Page
    customers: {
      title: 'Customers',
      addCustomer: 'Add Customer',
      searchPlaceholder: 'Search customers...',
      name: 'Name',
      email: 'Email',
      phone: 'Phone',
      type: 'Type',
      status: 'Status',
      actions: 'Actions',
      viewDetails: 'View Details',
      edit: 'Edit',
      delete: 'Delete',
      regular: 'Regular',
      b2b: 'B2B',
      active: 'Active',
      inactive: 'Inactive',
      createNew: 'Create New Customer',
      personalInfo: 'Personal Information',
      firstName: 'First Name',
      lastName: 'Last Name',
      companyInfo: 'Company Information',
      companyName: 'Company Name',
      rneNumber: 'RNE Number',
      commercialRegistration: 'Commercial Registration Number',
      taxId: 'Tax ID',
      customerType: 'Customer Type',
      selectType: 'Select type...',
      uploadRne: 'Upload RNE PDF',
      clickToUpload: 'Click to upload RNE PDF',
      pdfUpTo: 'PDF up to 5MB',
      financialSettings: 'Financial Settings (B2B)',
      configurePayment: 'Configure payment terms and credit limits for this customer',
      paymentMethod: 'Payment Method',
      cashOnDelivery: 'Cash on Delivery',
      creditLimit: 'Credit Limit (TND)',
      maxOutstanding: 'Maximum outstanding balance allowed for this customer',
      accountStatus: 'Account Status',
      commercialInProcess: 'Commercial In-Process',
      commercialInProcessDesc: 'Default for new customers',
      financialInProcess: 'Financial In-Process',
      financialInProcessDesc: 'Under financial review',
      activeStatus: 'Active',
      activeStatusDesc: 'Account fully operational',
      suspended: 'Suspended',
      suspendedDesc: 'Account temporarily disabled',
      financialNonCurrent: 'Financial Non-Current',
      financialNonCurrentDesc: 'Payment issues detected',
      noteOutstanding: 'Note',
      outstandingNote: 'Current Outstanding balance starts at 0 TND and will be automatically updated when quotations are approved and orders are placed.',
      loginCredentials: 'Login Credentials',
      username: 'Username',
      leaveEmpty: 'Leave empty to use email prefix',
      usernameGenerated: 'If empty, username will be generated from email',
      autoGeneratePassword: 'Auto-generate secure password',
      addressOptional: 'Address (Optional)',
      street: 'Street',
      city: 'City',
      stateRegion: 'State/Region',
      postalCode: 'Postal Code',
      country: 'Country',
      createCustomer: 'Create Customer',
      customerInfo: 'Customer Information',
      paymentTerms: 'Payment Terms',
      company: 'Company',
      selectPaymentTerms: 'Select payment terms...',
      defaultForNew: 'Default for new customers',
      underFinancialReview: 'Under financial review',
      canSubmitQuotations: 'Can submit quotations and place orders',
      accountTemporarilyDisabled: 'Account temporarily disabled',
      paymentIssuesDetected: 'Payment issues detected',
      enterPassword: 'Enter password',
      // Dropdown Options
      retailer: 'Retailer',
      wholesaler: 'Wholesaler',
      contractor: 'Contractor',
      industrial: 'Industrial',
      individual: 'Individual',
      other: 'Other',
      chequeOnDelivery: 'Cheque on Delivery',
      netTerms: 'Payment on Due Date (Net Terms)',
      net30: 'NET 30 (30 days)',
      net60: 'NET 60 (60 days)',
      net90: 'NET 90 (90 days)',
      net120: 'NET 120 (120 days)',
    },
    // Marketing Page
    marketing: {
      title: 'Marketing',
      comingSoon: 'Coming soon...',
    },
    // Accounting Page
    accounting: {
      title: 'Accounting',
      overview: 'Financial Overview',
      revenue: 'Revenue (HT)',
      tvaCollected: 'TVA Collected',
      estimatedProfit: 'Estimated Profit',
      margin: 'Margin',
      transactionCount: 'Transactions',
      monthlyStats: 'Monthly Statistics',
      period: 'Period',
      currency: 'TND',
    },
    // Settings Page
    settings: {
      title: 'Settings',
      subtitle: 'Manage your admin dashboard preferences',
      tabs: {
        general: 'General',
        notifications: 'Notifications',
        business: 'Business Info',
        system: 'System',
      },
      general: {
        title: 'General Settings',
        language: 'Language',
        languageDesc: 'Select your preferred language for the admin dashboard',
        english: 'English',
        french: 'Français',
      },
      notifications: {
        title: 'Notification Preferences',
        email: 'Email Notifications',
        emailDesc: 'Receive notifications via email',
        newOrders: 'New Orders',
        newOrdersDesc: 'Get notified when new orders are placed',
        newQuotations: 'New Quotations',
        newQuotationsDesc: 'Get notified when customers submit quotations',
        lowStock: 'Low Stock Alerts',
        lowStockDesc: 'Get notified when products are running low',
        systemUpdates: 'System Updates',
        systemUpdatesDesc: 'Receive notifications about system updates',
      },
      business: {
        title: 'Business Information',
        companyName: 'Company Name',
        email: 'Business Email',
        phone: 'Phone Number',
        address: 'Business Address',
        taxId: 'Tax ID / Registration Number',
        currency: 'Currency',
      },
      system: {
        title: 'System Settings',
        maintenance: 'Maintenance Mode',
        maintenanceDesc: 'Enable maintenance mode to prevent customer access',
        autoBackup: 'Automatic Backups',
        autoBackupDesc: 'Automatically backup database daily',
        sessionTimeout: 'Session Timeout (minutes)',
        sessionTimeoutDesc: 'Auto-logout after inactivity',
      },
      buttons: {
        save: 'Save Changes',
        saved: 'Saved Successfully!',
      },
    },
    // Common
    common: {
      save: 'Save',
      cancel: 'Cancel',
      delete: 'Delete',
      edit: 'Edit',
      view: 'View',
      search: 'Search',
      filter: 'Filter',
      export: 'Export',
      import: 'Import',
      loading: 'Loading...',
      noData: 'No data available',
      confirm: 'Confirm',
      yes: 'Yes',
      no: 'No',
      close: 'Close',
    },
  },
  fr: {
    // Sidebar Navigation
    nav: {
      dashboard: 'Tableau de bord',
      products: 'Produits',
      categories: 'Catégories',
      catalogs: 'Catalogues',
      orders: 'Commandes',
      quotations: 'Devis',
      customers: 'Clients',
      marketing: 'Marketing',
      accounting: 'Comptabilité',
      settings: 'Paramètres',
      logout: 'Déconnexion',
    },
    // Dashboard Page
    dashboard: {
      title: 'Tableau de bord',
      welcome: 'Bienvenue',
      overview: 'Aperçu',
      totalRevenue: 'Revenu Total',
      totalOrders: 'Total Commandes',
      totalCustomers: 'Total Clients',
      pendingQuotations: 'Devis en Attente',
      recentOrders: 'Commandes Récentes',
      recentQuotations: 'Devis Récents',
      viewAll: 'Voir Tout',
      orderNumber: 'Commande',
      customer: 'Client',
      amount: 'Montant',
      status: 'Statut',
      date: 'Date',
      actions: 'Actions',
      viewDetails: 'Voir Détails',
      noOrders: 'Aucune commande',
      noQuotations: 'Aucun devis',
    },
    // Products Page
    products: {
      title: 'Produits',
      addProduct: 'Ajouter Produit',
      searchPlaceholder: 'Rechercher des produits...',
      name: 'Nom',
      category: 'Catégorie',
      price: 'Prix',
      stock: 'Stock',
      status: 'Statut',
      actions: 'Actions',
      edit: 'Modifier',
      delete: 'Supprimer',
      active: 'Actif',
      inactive: 'Inactif',
      inStock: 'En Stock',
      outOfStock: 'Rupture de Stock',
      lowStock: 'Stock Bas',
      // Product Creation/Edit Form
      createNew: 'Créer Nouveau Produit',
      editProduct: 'Modifier Produit',
      addNewProduct: 'Ajouter un nouveau produit à votre catalogue',
      updateProductInfo: 'Mettre à jour les informations du produit',
      preview: 'Aperçu',
      saveAsDraft: 'Enregistrer comme Brouillon',
      createProduct: 'Créer Produit',
      updateProduct: 'Mettre à Jour Produit',
      saving: 'Enregistrement...',
      updating: 'Mise à jour...',
      // Tabs
      productInfo: 'Infos Produit',
      measurementSelling: 'Mesure & Vente',
      sizesSpecs: 'Tailles & Spécifications',
      pricingInventory: 'Prix & Inventaire',
      seo: 'SEO',
      // Product Info Section
      productName: 'Nom du Produit',
      productNamePlaceholder: 'ex., Perceuse Sans Fil 18V',
      selectCategory: 'Sélectionner une catégorie',
      description: 'Description',
      descriptionPlaceholder: 'Description détaillée du produit...',
      characters: 'caractères',
      productImages: 'Images du Produit',
      uploadImage: 'Télécharger image',
      uploadFile: 'Télécharger Fichier',
      useUrl: 'Utiliser URL',
      firstImagePrimary: 'La première image sera l\'image principale. Taille recommandée: 800x800px',
      brand: 'Marque',
      brandPlaceholder: 'ex., Bosch, DeWalt, Makita',
      productStatus: 'Statut du Produit',
      draft: 'Brouillon',
      hidden: 'Masqué',
      // Measurement Section
      sellingType: 'Type de Vente',
      piece: 'Pièce',
      weight: 'Poids',
      length: 'Longueur',
      volume: 'Volume',
      custom: 'Personnalisé',
      customUnit: 'Unité Personnalisée',
      customUnitPlaceholder: 'ex., boîte, rouleau, feuille',
      packSizes: 'Tailles de Pack',
      packSizesDesc: 'Définir différentes quantités de pack pour ce produit',
      addPackSize: 'Ajouter Taille de Pack',
      quantity: 'Quantité',
      label: 'Étiquette',
      // Size Specifications
      hasSizes: 'Ce produit a plusieurs tailles',
      sizeVariations: 'Variations de Taille',
      sizeVariationsDesc: 'Définir différentes tailles et leurs spécifications',
      addSize: 'Ajouter Taille',
      sizeName: 'Nom de Taille',
      dimension: 'Dimension',
      // Pricing & Inventory
      basePrice: 'Prix de Base',
      sku: 'SKU',
      skuPlaceholder: 'ex., DRL-18V-001',
      stockQuantity: 'Quantité en Stock',
      lowStockThreshold: 'Seuil de Stock Bas',
      lowStockThresholdDesc: 'Être notifié quand le stock tombe en dessous de ce nombre',
      // SEO Section
      metaTitle: 'Méta Titre',
      metaTitlePlaceholder: 'Titre optimisé SEO',
      metaDescription: 'Méta Description',
      metaDescriptionPlaceholder: 'Description optimisée SEO',
      slug: 'URL Slug',
      slugPlaceholder: 'url-slug-produit',
      // Navigation
      previous: 'Précédent',
      next: 'Suivant',
      // Validation
      productNameRequired: 'Le nom du produit est requis',
      categoryRequired: 'La catégorie est requise',
      basePriceRequired: 'Le prix de base doit être supérieur à 0',
      skuRequired: 'Le SKU est requis',
      fillRequiredFields: 'Veuillez remplir tous les champs requis',
      selectCategoryFirst: 'Veuillez sélectionner une catégorie',
      enterValidPrice: 'Veuillez entrer un prix valide',
      // Success Messages
      productCreated: 'Produit créé avec succès!',
      productUpdated: 'Produit mis à jour avec succès!',
      productSavedDraft: 'Produit enregistré comme brouillon avec succès!',
      // Error Messages
      failedToCreate: 'Échec de la création du produit',
      failedToUpdate: 'Échec de la mise à jour du produit',
      failedToLoad: 'Échec du chargement du produit',
    },
    // Categories Page
    categories: {
      title: 'Catégories',
      addCategory: 'Ajouter Catégorie',
      name: 'Nom',
      products: 'Produits',
      status: 'Statut',
      actions: 'Actions',
      edit: 'Modifier',
      delete: 'Supprimer',
    },
    // Orders Page
    orders: {
      title: 'Commandes',
      orderNumber: 'Numéro de Commande',
      customer: 'Client',
      date: 'Date',
      total: 'Total',
      status: 'Statut',
      actions: 'Actions',
      viewDetails: 'Voir Détails',
      filterByStatus: 'Filtrer par Statut',
      allStatuses: 'Tous les Statuts',
      pending: 'En Attente',
      processing: 'En Cours',
      shipped: 'Expédié',
      delivered: 'Livré',
      cancelled: 'Annulé',
      editOrder: 'Modifier Commande',
      modifyItems: 'Modifier les articles avant de changer le statut',
      addProduct: 'Ajouter Produit',
      searchProducts: 'Rechercher des produits par nom ou SKU...',
      orderItems: 'Articles de Commande',
      changeFromOriginal: 'Changement par rapport à l\'original',
      saveChanges: 'Enregistrer les Modifications',
    },
    // Quotations Page
    quotations: {
      title: 'Devis',
      quotationNumber: 'Numéro de Devis',
      customer: 'Client',
      date: 'Date',
      total: 'Total',
      status: 'Statut',
      actions: 'Actions',
      viewDetails: 'Voir Détails',
      approve: 'Approuver',
      decline: 'Refuser',
      pendingApproval: 'En Attente d\'Approbation',
      approved: 'Approuvé',
      declined: 'Refusé',
      filterByStatus: 'Filtrer par Statut',
      allStatuses: 'Tous les Statuts',
      convertedToOrder: 'Converti en Commande',
      quotationWillExceed: 'Ce devis dépassera la limite de crédit du client de',
      creditLimit: 'Limite de Crédit',
      currentOutstanding: 'Solde Impayé Actuel',
      newOutstanding: 'Nouveau Solde Impayé',
      overLimitBy: 'Dépassement de',
      customerInformation: 'Informations Client',
      accountStatus: 'Statut du Compte',
      items: 'Articles',
      sku: 'SKU',
      quantity: 'Quantité',
      unitPrice: 'Prix Unitaire',
      financialSummary: 'Résumé Financier',
      subtotal: 'Sous-total',
      tax: 'Taxe',
      created: 'Créé',
      availableAfter: 'Disponible Après',
      creditLimitExceeded: 'Ce devis dépassera la limite de crédit du client de {amount}',
      withinCreditLimit: 'Dans la limite de crédit. Disponible après: {amount}',
      confirmApprove: 'Êtes-vous sûr de vouloir approuver ce devis?',
      approveFailed: 'Échec de l\'approbation du devis',
      declineReasonRequired: 'Veuillez fournir une raison pour le refus',
      declineFailed: 'Échec du refus du devis',
    },
    // Customers Page
    customers: {
      title: 'Clients',
      addCustomer: 'Ajouter Client',
      searchPlaceholder: 'Rechercher des clients...',
      name: 'Nom',
      email: 'Email',
      phone: 'Téléphone',
      type: 'Type',
      status: 'Statut',
      actions: 'Actions',
      viewDetails: 'Voir Détails',
      edit: 'Modifier',
      delete: 'Supprimer',
      regular: 'Régulier',
      b2b: 'B2B',
      active: 'Actif',
      inactive: 'Inactif',
      createNew: 'Créer Nouveau Client',
      personalInfo: 'Informations Personnelles',
      firstName: 'Prénom',
      lastName: 'Nom',
      companyInfo: 'Informations Entreprise',
      companyName: 'Nom de l\'Entreprise',
      rneNumber: 'Numéro RNE',
      commercialRegistration: 'Numéro d\'Immatriculation Commerciale',
      taxId: 'Identifiant Fiscal',
      customerType: 'Type de Client',
      selectType: 'Sélectionner type...',
      uploadRne: 'Télécharger RNE PDF',
      clickToUpload: 'Cliquer pour télécharger RNE PDF',
      pdfUpTo: 'PDF jusqu\'à 5MB',
      financialSettings: 'Paramètres Financiers (B2B)',
      configurePayment: 'Configurer les conditions de paiement et limites de crédit pour ce client',
      paymentMethod: 'Méthode de Paiement',
      cashOnDelivery: 'Paiement à la Livraison',
      creditLimit: 'Limite de Crédit (TND)',
      maxOutstanding: 'Solde impayé maximum autorisé pour ce client',
      accountStatus: 'Statut du Compte',
      commercialInProcess: 'Commercial En Cours',
      commercialInProcessDesc: 'Par défaut pour nouveaux clients',
      financialInProcess: 'Financier En Cours',
      financialInProcessDesc: 'Sous révision financière',
      activeStatus: 'Actif',
      activeStatusDesc: 'Compte entièrement opérationnel',
      suspended: 'Suspendu',
      suspendedDesc: 'Compte temporairement désactivé',
      financialNonCurrent: 'Financier Non-Courant',
      financialNonCurrentDesc: 'Problèmes de paiement détectés',
      noteOutstanding: 'Note',
      outstandingNote: 'Le solde impayé actuel commence à 0 TND et sera automatiquement mis à jour lorsque les devis sont approuvés et les commandes sont passées.',
      loginCredentials: 'Identifiants de Connexion',
      username: 'Nom d\'utilisateur',
      leaveEmpty: 'Laisser vide pour utiliser le préfixe email',
      usernameGenerated: 'Si vide, le nom d\'utilisateur sera généré à partir de l\'email',
      autoGeneratePassword: 'Générer automatiquement un mot de passe sécurisé',
      addressOptional: 'Adresse (Optionnel)',
      street: 'Rue',
      city: 'Ville',
      stateRegion: 'État/Région',
      postalCode: 'Code Postal',
      country: 'Pays',
      createCustomer: 'Créer Client',
      customerInfo: 'Informations Client',
      paymentTerms: 'Conditions de Paiement',
      company: 'Entreprise',
      selectPaymentTerms: 'Sélectionner conditions de paiement...',
      defaultForNew: 'Par défaut pour nouveaux clients',
      underFinancialReview: 'Sous révision financière',
      canSubmitQuotations: 'Peut soumettre des devis et passer des commandes',
      accountTemporarilyDisabled: 'Compte temporairement désactivé',
      paymentIssuesDetected: 'Problèmes de paiement détectés',
      enterPassword: 'Entrer le mot de passe',
      // Dropdown Options
      retailer: 'Détaillant',
      wholesaler: 'Grossiste',
      contractor: 'Entrepreneur',
      industrial: 'Industriel',
      individual: 'Particulier',
      other: 'Autre',
      chequeOnDelivery: 'Chèque à la livraison',
      netTerms: 'Paiement à échéance',
      net30: 'NET 30 (30 jours)',
      net60: 'NET 60 (60 jours)',
      net90: 'NET 90 (90 jours)',
      net120: 'NET 120 (120 jours)',
    },
    // Marketing Page
    marketing: {
      title: 'Marketing',
      comingSoon: 'Bientôt disponible...',
    },
    // Accounting Page
    accounting: {
      title: 'Comptabilité',
      overview: 'Aperçu Financier',
      revenue: 'Chiffre d\'Affaires (HT)',
      revenueTTC: 'Chiffre d\'Affaires (TTC)',
      tvaCollected: 'TVA Collectée',
      tvaBreakdown: 'Détail TVA',
      timbreFiscal: 'Timbre Fiscal',
      estimatedProfit: 'Bénéfice Estimé',
      totalCost: 'Coût Total',
      grossMargin: 'Marge Brute',
      margin: 'Marge',
      transactionCount: 'Transactions',
      monthlyStats: 'Statistiques Mensuelles',
      period: 'Période',
      currency: 'TND',
      volume: 'Volume',
      invoices: 'Factures',
      orders: 'Commandes',
      averageBasket: 'Panier Moyen',
      today: 'Aujourd\'hui',
      thisMonth: 'Ce Mois',
      lastMonth: 'Mois Dernier',
      thisYear: 'Cette Année',
      custom: 'Personnalisé',
      rate: 'Taux',
      amount: 'Montant',
    },
    // Settings Page
    settings: {
      title: 'Paramètres',
      subtitle: 'Gérez vos préférences du tableau de bord',
      tabs: {
        general: 'Général',
        notifications: 'Notifications',
        business: 'Infos Entreprise',
        system: 'Système',
      },
      general: {
        title: 'Paramètres Généraux',
        language: 'Langue',
        languageDesc: 'Sélectionnez votre langue préférée pour le tableau de bord',
        english: 'English',
        french: 'Français',
      },
      notifications: {
        title: 'Préférences de Notification',
        email: 'Notifications Email',
        emailDesc: 'Recevoir des notifications par email',
        newOrders: 'Nouvelles Commandes',
        newOrdersDesc: 'Être notifié lors de nouvelles commandes',
        newQuotations: 'Nouveaux Devis',
        newQuotationsDesc: 'Être notifié lors de soumissions de devis',
        lowStock: 'Alertes Stock Bas',
        lowStockDesc: 'Être notifié quand les produits sont en rupture',
        systemUpdates: 'Mises à Jour Système',
        systemUpdatesDesc: 'Recevoir des notifications sur les mises à jour',
      },
      business: {
        title: 'Informations Entreprise',
        companyName: 'Nom de l\'Entreprise',
        email: 'Email Professionnel',
        phone: 'Numéro de Téléphone',
        address: 'Adresse Professionnelle',
        taxId: 'Numéro Fiscal / Immatriculation',
        currency: 'Devise',
      },
      system: {
        title: 'Paramètres Système',
        maintenance: 'Mode Maintenance',
        maintenanceDesc: 'Activer le mode maintenance pour bloquer l\'accès client',
        autoBackup: 'Sauvegardes Automatiques',
        autoBackupDesc: 'Sauvegarder automatiquement la base de données quotidiennement',
        sessionTimeout: 'Délai de Session (minutes)',
        sessionTimeoutDesc: 'Déconnexion automatique après inactivité',
      },
      buttons: {
        save: 'Enregistrer les Modifications',
        saved: 'Enregistré avec Succès!',
      },
    },
    // Common
    common: {
      save: 'Enregistrer',
      cancel: 'Annuler',
      delete: 'Supprimer',
      edit: 'Modifier',
      view: 'Voir',
      search: 'Rechercher',
      filter: 'Filtrer',
      export: 'Exporter',
      import: 'Importer',
      loading: 'Chargement...',
      noData: 'Aucune donnée disponible',
      confirm: 'Confirmer',
      yes: 'Oui',
      no: 'Non',
      close: 'Fermer',
    },
  },
};

export function LanguageProvider({ children }: { children: ReactNode }) {
  const { user } = useAuth();
  const [language, setLanguageState] = useState<Language>(() => {
    const saved = localStorage.getItem('adminLanguage');
    return (saved as Language) || 'en';
  });

  // Sync with user preference when user logs in
  useEffect(() => {
    if (user?.language && (user.language === 'en' || user.language === 'fr')) {
      if (user.language !== language) {
        setLanguageState(user.language as Language);
        localStorage.setItem('adminLanguage', user.language);
      }
    }
  }, [user]);

  const setLanguage = async (lang: Language) => {
    setLanguageState(lang);
    localStorage.setItem('adminLanguage', lang);

    // Persist to backend if user is logged in
    if (user) {
      try {
        const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:3001/api';
        await fetch(`${API_URL}/auth/me/language`, {
          method: 'PATCH',
          headers: buildAuthHeaders(),
          body: JSON.stringify({ language: lang }),
        });
      } catch (error) {
        console.error('Failed to persist language preference:', error);
      }
    }
  };

  const value = {
    language,
    setLanguage,
    t: translations[language],
  };

  return (
    <LanguageContext.Provider value={value}>
      {children}
    </LanguageContext.Provider>
  );
}

export function useLanguage() {
  const context = useContext(LanguageContext);
  if (context === undefined) {
    throw new Error('useLanguage must be used within a LanguageProvider');
  }
  return context;
}
