import { useState, useEffect } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { ArrowLeft, Loader } from 'lucide-react';
import QuotationDocument from '../components/QuotationDocument';
import { quotationService } from '../services/quotationService';

export default function QuotationDetailScreen() {
    const { id } = useParams<{ id: string }>();
    const navigate = useNavigate();
    const [quotation, setQuotation] = useState<any>(null);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState<string | null>(null);

    useEffect(() => {
        if (id) {
            loadQuotation();
        }
    }, [id]);

    const loadQuotation = async () => {
        try {
            setLoading(true);
            const data = await quotationService.getById(id!);
            setQuotation(data);
        } catch (err: any) {
            console.error('Error loading quotation:', err);
            setError(err.message || 'Failed to load quotation');
        } finally {
            setLoading(false);
        }
    };

    if (loading) {
        return (
            <div className="flex items-center justify-center min-h-screen">
                <Loader className="w-8 h-8 animate-spin text-blue-600" />
            </div>
        );
    }

    if (error || !quotation) {
        return (
            <div className="flex flex-col items-center justify-center min-h-screen p-4">
                <div className="text-center">
                    <h2 className="text-xl font-semibold text-gray-800 mb-2">
                        Erreur de chargement
                    </h2>
                    <p className="text-gray-600 mb-4">{error || 'Devis introuvable'}</p>
                    <button
                        onClick={() => navigate('/my-quotations')}
                        className="bg-blue-600 text-white px-6 py-2 rounded hover:bg-blue-700"
                    >
                        Retour aux devis
                    </button>
                </div>
            </div>
        );
    }

    return (
        <div className="min-h-screen bg-gray-50">
            {/* Header with back button */}
            <div className="bg-white border-b border-gray-200 p-4 no-print">
                <div className="max-w-[210mm] mx-auto flex items-center gap-4">
                    <button
                        onClick={() => navigate('/my-quotations')}
                        className="flex items-center gap-2 text-gray-600 hover:text-gray-900"
                    >
                        <ArrowLeft className="w-5 h-5" />
                        Retour
                    </button>
                    <h1 className="text-xl font-semibold">
                        Devis Nº {quotation.quotationNumber}
                    </h1>
                    <div className="ml-auto">
                        <span className={`px-3 py-1 rounded-full text-sm font-medium ${quotation.status === 'APPROVED' ? 'bg-green-100 text-green-700' :
                                quotation.status === 'REJECTED' ? 'bg-red-100 text-red-700' :
                                    'bg-yellow-100 text-yellow-700'
                            }`}>
                            {quotation.status === 'APPROVED' ? 'Approuvé' :
                                quotation.status === 'REJECTED' ? 'Rejeté' :
                                    'En attente'}
                        </span>
                    </div>
                </div>
            </div>

            {/* Document */}
            <div className="py-8">
                <QuotationDocument quotation={quotation} />
            </div>
        </div>
    );
}
