import { Menu, ShoppingCart, ArrowLeft, User, FileText, Receipt, Home } from 'lucide-react';
import { useLanguage } from '../contexts/LanguageContext';
import { Category } from '../types/api';
import GlobalSearch from './GlobalSearch';
import { Link, useLocation } from 'react-router-dom';

interface HeaderProps {
  title?: string;
  showBack?: boolean;
  onBackClick?: () => void;
  onMenuClick: () => void;
  cartItemCount: number;
  onCartClick: () => void;
  onAccountClick?: () => void;
  isAuthenticated?: boolean;
  categories?: Category[];
}

export default function Header({
  // title = 'SMD Hardware Store', // Unused
  showBack = false,
  onBackClick,
  onMenuClick,
  cartItemCount,
  onCartClick,
  onAccountClick,
  isAuthenticated = false,
  categories = []
}: HeaderProps) {
  const { t } = useLanguage();
  const location = useLocation();

  const isActive = (path: string) => location.pathname === path;

  return (
    <header className="bg-white shadow-sm sticky top-0 z-40 border-b border-gray-100">
      <div className="max-w-7xl mx-auto">
        {/* Top Bar - Logo, Search, Actions */}
        <div className="flex items-center justify-between px-4 py-3 gap-4">
          {/* Left: Logo & Mobile Menu */}
          <div className="flex items-center gap-3 flex-shrink-0">
            {showBack ? (
              <button
                onClick={onBackClick}
                className="p-2 -ml-2 hover:bg-gray-100 rounded-lg transition-colors"
              >
                <ArrowLeft className="w-6 h-6 text-gray-700" />
              </button>
            ) : (
              <button
                onClick={onMenuClick}
                className="p-2 -ml-2 hover:bg-gray-100 rounded-lg transition-colors lg:hidden"
              >
                <Menu className="w-6 h-6 text-gray-700" />
              </button>
            )}

            <Link to="/" className="flex items-center gap-2 hover:opacity-90 transition-opacity">
              <img src="/logo.png" alt="SMD Logo" className="h-10 w-auto object-contain" />
              <span className="text-lg font-bold text-gray-900 hidden sm:block">
                SMD
              </span>
            </Link>
          </div>

          {/* Center: Global Search (Desktop) */}
          <div className="hidden md:block flex-1 max-w-2xl px-8">
            <GlobalSearch categories={categories} />
          </div>

          {/* Right: Actions */}
          <div className="flex items-center gap-2 flex-shrink-0">
            {/* Mobile Search Toggle (could be added here if needed, but GlobalSearch is responsive enough to fit or stack) */}

            {onAccountClick && (
              <button
                onClick={onAccountClick}
                className={`relative p-2.5 rounded-lg transition-colors ${isActive('/account') ? 'bg-blue-50 text-blue-600' : 'hover:bg-gray-100 text-gray-700'
                  }`}
                aria-label="Account"
              >
                <User className="w-6 h-6" />
                {isAuthenticated && (
                  <span className="absolute top-2 right-2 bg-green-500 w-2 h-2 rounded-full border-2 border-white"></span>
                )}
              </button>
            )}

            <button
              onClick={onCartClick}
              className={`relative p-2.5 rounded-lg transition-colors ${isActive('/basket') ? 'bg-blue-50 text-blue-600' : 'hover:bg-gray-100 text-gray-700'
                }`}
              aria-label="Shopping Cart"
            >
              <ShoppingCart className="w-6 h-6" />
              {cartItemCount > 0 && (
                <span className="absolute -top-1 -right-1 bg-orange-500 text-white text-xs font-bold rounded-full min-w-[20px] h-5 flex items-center justify-center px-1 border-2 border-white">
                  {cartItemCount > 9 ? '9+' : cartItemCount}
                </span>
              )}
            </button>
          </div>
        </div>

        {/* Mobile Search (below header on small screens) */}
        <div className="md:hidden px-4 pb-3">
          <GlobalSearch categories={categories} />
        </div>

        {/* Navigation Links (Desktop) */}
        <div className="hidden md:flex items-center px-4 border-t border-gray-100 overflow-x-auto">
          <Link
            to="/"
            className={`flex items-center gap-2 px-4 py-3 text-sm font-medium border-b-2 transition-colors whitespace-nowrap ${isActive('/') ? 'border-blue-600 text-blue-600' : 'border-transparent text-gray-600 hover:text-gray-900'
              }`}
          >
            <Home className="w-4 h-4" />
            {t.nav?.shop || 'Shop'}
          </Link>
          {isAuthenticated && (
            <>
              <Link
                to="/quotations"
                className={`flex items-center gap-2 px-4 py-3 text-sm font-medium border-b-2 transition-colors whitespace-nowrap ${isActive('/quotations') ? 'border-blue-600 text-blue-600' : 'border-transparent text-gray-600 hover:text-gray-900'
                  }`}
              >
                <FileText className="w-4 h-4" />
                {t.nav?.quotations || 'My Quotations'}
              </Link>
              <Link
                to="/invoices"
                className={`flex items-center gap-2 px-4 py-3 text-sm font-medium border-b-2 transition-colors whitespace-nowrap ${isActive('/invoices') ? 'border-blue-600 text-blue-600' : 'border-transparent text-gray-600 hover:text-gray-900'
                  }`}
              >
                <Receipt className="w-4 h-4" />
                {t.nav?.invoices || 'My Invoices'}
              </Link>
            </>
          )}
        </div>
      </div>
    </header>
  );
}
