import { ReactNode, useEffect, useMemo, useState } from 'react';
import { Routes, Route, Navigate, useNavigate, useLocation, useParams, BrowserRouter } from 'react-router-dom';

import { CartProvider, useCart, CartItem } from './context/CartContext';
import { AuthProvider, useAuth } from './context/AuthContext';
import { LanguageProvider } from './contexts/LanguageContext';
import Header from './components/Header';
import Sidebar from './components/Sidebar';
import HomeScreen from './screens/HomeScreen';
import ProductListScreen from './screens/ProductListScreen';
import BasketScreen from './screens/BasketScreen';
import SubcategoryScreen from './screens/SubcategoryScreen';
import ProductDetailScreen from './screens/ProductDetailScreen';
import CheckoutScreen from './screens/CheckoutScreen';
import OrderConfirmationScreen from './screens/OrderConfirmationScreen';
import LoginScreen from './screens/LoginScreen';
import AccountScreen from './screens/AccountScreen';
import MyQuotationsScreen from './screens/MyQuotationsScreen';
import QuotationDetailScreen from './screens/QuotationDetailScreen';
import MyInvoicesScreen from './screens/MyInvoicesScreen';
import { API } from './services/api';
import { Category } from './types/api';

const ROUTE_TITLES: Record<string, string> = {
  '/': 'Hardware Store',
  '/basket': 'Shopping Basket',
  '/checkout': 'Checkout',
  '/orders/confirmation': 'Order Confirmed',
  '/login': 'Customer Login',
  '/account': 'My Account',
  '/quotations': 'My Quotations',
  '/invoices': 'My Invoices',
};


import PWAInstallPopup from './components/PWAInstallPopup';

export default function App() {
  return (
    <BrowserRouter>
      <LanguageProvider>
        <AuthProvider>
          <CartProvider>
            <AppRoutes />
            <PWAInstallPopup />
          </CartProvider>
        </AuthProvider>
      </LanguageProvider>
    </BrowserRouter>
  );
}


function AppRoutes() {
  const navigate = useNavigate();
  const location = useLocation();
  const [sidebarOpen, setSidebarOpen] = useState(false);
  const [categories, setCategories] = useState<Category[]>([]);
  const [categoryError, setCategoryError] = useState<string | null>(null);
  const [categoriesLoading, setCategoriesLoading] = useState(true);
  const [homeSearchQuery, setHomeSearchQuery] = useState('');
  const [productSearchQuery, setProductSearchQuery] = useState('');
  const [subcategorySearchQuery, setSubcategorySearchQuery] = useState('');
  const [orderData, setOrderData] = useState<any>(null);
  const { cartItems, addToCart, removeFromCart, updateQuantity, getTotalItems, isLoading: cartLoading, debugClearCart, debugLogCart, clearCart } = useCart();
  const { isAuthenticated, isLoading: authLoading, catalogId } = useAuth();

  useEffect(() => {
    (window as any).cartDebug = {
      clear: debugClearCart,
      log: debugLogCart,
      items: cartItems,
      count: cartItems.length,
    };
  }, [cartItems, debugClearCart, debugLogCart]);

  useEffect(() => {
    const loadCategories = async () => {
      try {
        setCategoriesLoading(true);
        const allCategories = await API.getAllCategories();
        setCategories(allCategories);
      } catch (err) {
        console.error('Error loading categories:', err);
        setCategoryError('Failed to load categories');
      } finally {
        setCategoriesLoading(false);
      }
    };

    loadCategories();
  }, [catalogId]);

  const rootCategories = useMemo(() => {
    return categories.filter((category) => !category.parentId);
  }, [categories]);

  const filteredCategories = useMemo(() => {
    if (!homeSearchQuery.trim()) {
      return rootCategories;
    }

    const query = homeSearchQuery.toLowerCase();
    return rootCategories.filter((category) =>
      category.name.toLowerCase().includes(query) ||
      category.description?.toLowerCase().includes(query)
    );
  }, [rootCategories, homeSearchQuery]);

  const showHeader = location.pathname !== '/login';
  const showBackButton = location.pathname !== '/';

  const headerTitle = useMemo(() => {
    if (location.pathname.startsWith('/categories')) {
      return 'Browse Categories';
    }
    if (location.pathname.startsWith('/products')) {
      return 'Product Details';
    }
    return ROUTE_TITLES[location.pathname] || 'Hardware Store';
  }, [location.pathname]);

  const isAppLoading = authLoading || categoriesLoading;

  const handleOrderComplete = (order: any) => {
    setOrderData(order);
    clearCart();
    navigate('/orders/confirmation', { state: { order } });
  };

  const handleLoginSuccess = () => {
    const redirectPath = (location.state as any)?.from?.pathname || '/';
    navigate(redirectPath, { replace: true, state: undefined });
  };

  const handleSidebarNavigate = (path: string) => {
    setSidebarOpen(false);
    if (path === '/account' && !isAuthenticated) {
      navigate('/login');
      return;
    }
    navigate(path);
  };

  if (isAppLoading) {
    return (
      <div className="min-h-screen bg-gray-50 flex justify-center items-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  if (categoryError) {
    return (
      <div className="min-h-screen bg-gray-50 flex flex-col justify-center items-center">
        <p className="text-red-600 mb-4">{categoryError}</p>
        <button
          onClick={() => window.location.reload()}
          className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700"
        >
          Retry
        </button>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50">
      {showHeader && (
        <Header
          title={headerTitle}
          showBack={showBackButton}
          onBackClick={() => navigate(-1)}
          onMenuClick={() => setSidebarOpen(true)}
          cartItemCount={getTotalItems()}
          onCartClick={() => navigate('/basket')}
          onAccountClick={() => navigate(isAuthenticated ? '/account' : '/login')}
          isAuthenticated={isAuthenticated}
          categories={categories}
        />
      )}

      <Sidebar isOpen={sidebarOpen} onClose={() => setSidebarOpen(false)} onNavigate={handleSidebarNavigate} />

      <main className={!showHeader ? '' : 'max-w-7xl mx-auto'}>
        <Routes key={catalogId || 'public'}>
          <Route
            path="/"
            element={
              <HomeScreen
                categories={filteredCategories}
                searchQuery={homeSearchQuery}
                onSearchChange={setHomeSearchQuery}
                onCategoryClick={(id) => navigate(`/categories/${id}`)}
                onProductClick={(productId) => navigate(`/products/${productId}`)}
              />
            }
          />

          <Route
            path="/categories/:categoryId"
            element={
              <CategoryRouteView
                categories={categories}
                onAddToCart={addToCart}
                productSearchQuery={productSearchQuery}
                setProductSearchQuery={setProductSearchQuery}
                subcategorySearchQuery={subcategorySearchQuery}
                setSubcategorySearchQuery={setSubcategorySearchQuery}
              />
            }
          />

          <Route
            path="/products/:productId"
            element={<ProductRouteView onAddToCart={addToCart} />}
          />

          <Route
            path="/basket"
            element={
              <BasketRoute
                cartItems={cartItems}
                cartLoading={cartLoading}
                updateQuantity={updateQuantity}
                removeFromCart={removeFromCart}
                onCheckout={() => (isAuthenticated ? navigate('/checkout') : navigate('/login', { state: { from: location } }))}
              />
            }
          />

          <Route
            path="/checkout"
            element={
              <ProtectedRoute>
                <CheckoutScreen
                  cartItems={cartItems}
                  onBack={() => navigate('/basket')}
                  onOrderComplete={handleOrderComplete}
                  onUpdateQuantity={(cartItemKey, quantity) => updateQuantity(cartItemKey, quantity)}
                  onRemoveItem={(cartItemKey) => removeFromCart(cartItemKey)}
                />
              </ProtectedRoute>
            }
          />

          <Route
            path="/orders/confirmation"
            element={
              <ProtectedRoute>
                <OrderConfirmationRoute orderData={orderData} clearOrderData={() => setOrderData(null)} />
              </ProtectedRoute>
            }
          />

          <Route path="/login" element={<LoginScreen onLoginSuccess={handleLoginSuccess} />} />

          <Route
            path="/account"
            element={
              <ProtectedRoute>
                <AccountScreen
                  onLogout={() => navigate('/login')}
                  onNavigateToShop={() => navigate('/')}
                  onNavigateToQuotations={() => navigate('/quotations')}
                  onNavigateToInvoices={() => navigate('/invoices')}
                />
              </ProtectedRoute>
            }
          />

          <Route
            path="/quotations"
            element={
              <ProtectedRoute>
                <MyQuotationsScreen onBack={() => navigate('/account')} />
              </ProtectedRoute>
            }
          />

          <Route
            path="/quotations/:id"
            element={
              <ProtectedRoute>
                <QuotationDetailScreen />
              </ProtectedRoute>
            }
          />

          <Route
            path="/invoices"
            element={
              <ProtectedRoute>
                <MyInvoicesScreen onBack={() => navigate('/account')} />
              </ProtectedRoute>
            }
          />

          <Route path="*" element={<Navigate to="/" replace />} />
        </Routes>
      </main>
    </div>
  );
}

function ProtectedRoute({ children }: { children: ReactNode }) {
  const { isAuthenticated, isLoading } = useAuth();
  const location = useLocation();

  if (isLoading) {
    return (
      <div className="min-h-screen bg-gray-50 flex justify-center items-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  if (!isAuthenticated) {
    return <Navigate to="/login" replace state={{ from: location }} />;
  }

  return <>{children}</>;
}

function CategoryRouteView({
  categories,
  onAddToCart,
  productSearchQuery,
  setProductSearchQuery,
  subcategorySearchQuery,
  setSubcategorySearchQuery,
}: {
  categories: Category[];
  onAddToCart: ReturnType<typeof useCart>['addToCart'];
  productSearchQuery: string;
  setProductSearchQuery: (value: string) => void;
  subcategorySearchQuery: string;
  setSubcategorySearchQuery: (value: string) => void;
}) {
  const { categoryId } = useParams();
  const navigate = useNavigate();
  const [category, setCategory] = useState<Category | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    const loadCategory = async () => {
      if (!categoryId) return;

      const cachedCategory = findCategoryById(categories, categoryId);
      if (cachedCategory) {
        setCategory(cachedCategory);
        setLoading(false);
        return;
      }

      try {
        setLoading(true);
        const fetchedCategory = await API.getCategory(categoryId);
        setCategory(fetchedCategory);
      } catch (err) {
        console.error('Error loading category:', err);
        setError('Category not found');
      } finally {
        setLoading(false);
      }
    };

    loadCategory();
  }, [categoryId, categories]);

  if (loading) {
    return (
      <div className="min-h-[50vh] flex justify-center items-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  if (error || !category) {
    return <Navigate to="/" replace />;
  }

  if (category.hasChildren) {
    return (
      <SubcategoryScreen
        parentCategory={category}
        onSubcategoryClick={(subCategory) => navigate(`/categories/${subCategory.id}`)}
        onProductClick={(productId) => navigate(`/products/${productId}`)}
        onAddToCart={onAddToCart}
        searchQuery={subcategorySearchQuery}
        onSearchChange={setSubcategorySearchQuery}
      />
    );
  }

  return (
    <ProductListScreen
      categoryId={category.id}
      searchQuery={productSearchQuery}
      onProductClick={(productId) => navigate(`/products/${productId}`)}
      onAddToCart={onAddToCart}
      onSearchChange={setProductSearchQuery}
    />
  );
}

function ProductRouteView({ onAddToCart }: { onAddToCart: ReturnType<typeof useCart>['addToCart'] }) {
  const { productId } = useParams();

  if (!productId) {
    return <Navigate to="/" replace />;
  }

  return <ProductDetailScreen productId={productId} onAddToCart={onAddToCart} />;
}

function OrderConfirmationRoute({ orderData, clearOrderData }: { orderData: any; clearOrderData: () => void }) {
  const location = useLocation();
  const navigate = useNavigate();
  const order = (location.state as any)?.order || orderData;

  useEffect(() => {
    if (!order) {
      navigate('/', { replace: true });
    }
  }, [order, navigate]);

  if (!order) {
    return null;
  }

  return (
    <OrderConfirmationScreen
      order={order}
      onBackToHome={() => {
        clearOrderData();
        navigate('/');
      }}
    />
  );
}

function BasketRoute({
  cartItems,
  cartLoading,
  updateQuantity,
  removeFromCart,
  onCheckout,
}: {
  cartItems: CartItem[];
  cartLoading: boolean;
  updateQuantity: (cartItemKey: string, quantity: number) => void;
  removeFromCart: (cartItemKey: string) => void;
  onCheckout: () => void;
}) {
  const handleUpdateQuantity = (productId: string, quantity: number) => {
    const item = cartItems.find((cartItem) => cartItem.product.id === productId);
    if (!item) return;
    updateQuantity(item.cartItemKey, quantity);
  };

  const handleRemoveItem = (productId: string) => {
    const item = cartItems.find((cartItem) => cartItem.product.id === productId);
    if (!item) return;
    removeFromCart(item.cartItemKey);
  };

  return (
    <BasketScreen
      cartItems={cartItems}
      onUpdateQuantity={handleUpdateQuantity}
      onRemoveItem={handleRemoveItem}
      isLoading={cartLoading}
      onCheckout={onCheckout}
    />
  );
}

function findCategoryById(categories: Category[], id: string): Category | null {
  for (const category of categories) {
    if (category.id === id) {
      return category;
    }

    if (category.children) {
      const found = findCategoryById(category.children, id);
      if (found) {
        return found;
      }
    }
  }

  return null;
}
