// Cart Debug Script - Paste this in browser console to debug cart issues

(function() {
  'use strict';
  
  const CART_KEY = 'hardware-store-cart';
  
  // Clear cart from localStorage
  function clearCart() {
    localStorage.removeItem(CART_KEY);
    console.log('✅ Cart cleared from localStorage');
    console.log('🔄 Refresh the page to see the changes');
  }
  
  // Get current cart
  function getCart() {
    const cart = localStorage.getItem(CART_KEY);
    return cart ? JSON.parse(cart) : [];
  }
  
  // Log cart contents
  function logCart() {
    const cart = getCart();
    console.log('=== Current Cart Contents ===');
    console.log('Number of items:', cart.length);
    console.log('Raw data:', cart);
    
    if (cart.length > 0) {
      cart.forEach((item, index) => {
        console.log(`Item ${index + 1}:`, {
          productId: item.product?.id,
          productName: item.product?.name,
          quantity: item.quantity,
          price: item.product?.price,
          cartItemKey: item.cartItemKey
        });
      });
    } else {
      console.log('Cart is empty');
    }
    console.log('===============================');
  }
  
  // Check for invalid items
  function validateCart() {
    const cart = getCart();
    const issues = [];
    
    if (!Array.isArray(cart)) {
      issues.push('Cart is not an array');
      return issues;
    }
    
    cart.forEach((item, index) => {
      if (!item.product || !item.product.id) {
        issues.push(`Item ${index}: Missing product data`);
      }
      if (!item.product || !item.product.name) {
        issues.push(`Item ${index}: Missing product name`);
      }
      if (!item.cartItemKey) {
        issues.push(`Item ${index}: Missing cartItemKey`);
      }
      if (typeof item.quantity !== 'number' || item.quantity <= 0) {
        issues.push(`Item ${index}: Invalid quantity`);
      }
    });
    
    if (issues.length > 0) {
      console.error('❌ Cart validation issues:', issues);
    } else {
      console.log('✅ Cart validation passed');
    }
    
    return issues;
  }
  
  // Auto-fix cart issues
  function fixCart() {
    const issues = validateCart();
    if (issues.length > 0) {
      console.log('🔧 Attempting to fix cart issues...');
      clearCart();
      console.log('✅ Cart cleared due to validation issues');
    } else {
      console.log('✅ No issues found, cart is valid');
    }
  }
  
  // Expose functions globally
  window.cartDebug = {
    clear: clearCart,
    get: getCart,
    log: logCart,
    validate: validateCart,
    fix: fixCart
  };
  
  console.log('🔧 Cart debug tools loaded!');
  console.log('Available commands:');
  console.log('  window.cartDebug.clear()  - Clear cart');
  console.log('  window.cartDebug.log()    - Show cart contents');
  console.log('  window.cartDebug.validate() - Check for issues');
  console.log('  window.cartDebug.fix()    - Auto-fix issues');
  console.log('  window.cartDebug.get()   - Get raw cart data');
  
})();
