/**
 * Core Hook System (Event Bus)
 * 
 * Implements a "Waterfull" or "Pipeline" pattern where plugins can intercept,
 * modify, or augment data flow.
 */

export type HookContext = any;
export type HookNextFunction = () => Promise<void>;
export type HookHandler = (context: HookContext, next: HookNextFunction) => Promise<void>;

export class HookSystem {
    private hooks: Map<string, HookHandler[]> = new Map();

    /**
     * Register a handler for a specific hook
     * @param hookName The name of the containment point (e.g. 'order.validate')
     * @param handler The function to execute
     */
    on(hookName: string, handler: HookHandler) {
        if (!this.hooks.has(hookName)) {
            this.hooks.set(hookName, []);
        }
        this.hooks.get(hookName)?.push(handler);
        console.log(`[HookSystem] Registered handler for '${hookName}'`);
    }

    /**
     * Trigger a hook. Handlers are executed sequentially.
     * @param hookName 
     * @param context The mutable context object passed through handlers
     * @returns The modified context
     */
    async trigger<T>(hookName: string, context: T): Promise<T> {
        const handlers = this.hooks.get(hookName) || [];

        if (handlers.length === 0) {
            return context;
        }

        // Execute handlers in sequence (Middleware pattern)
        // We compose them so that each handler calls 'next' to proceed
        const dispatch = async (index: number): Promise<void> => {
            if (index >= handlers.length) return;

            const handler = handlers[index];

            // [HARDENING] Crash Containment & Next Guard
            let nextCalled = false;
            const safeNext = async () => {
                if (nextCalled) {
                    console.warn(`[HookSystem] ⚠️ Warning: Handler for '${hookName}' called next() multiple times. Ignoring duplicate call.`);
                    return;
                }
                nextCalled = true;
                await dispatch(index + 1);
            };

            try {
                await handler(context, safeNext);
            } catch (error) {
                console.error(`[HookSystem] 🚨 CRASH: Plugin handler failed in '${hookName}'`, error);
                throw error; // Rethrow to maintain system behavior (500 Error) but ensure visibility
            }
        };

        await dispatch(0);
        return context;
    }
}

export const hooks = new HookSystem();
