import { PrismaClient } from '@prisma/client';

const prisma = new PrismaClient();

async function setDefaultCatalogAndFixNulls() {
    console.log('=== SETTING DEFAULT CATALOG & FIXING NULL VALUES ===\n');

    // Get the medical catalog (or whichever you want as default)
    const catalogs = await prisma.catalog.findMany();

    if (catalogs.length === 0) {
        console.error('❌ No catalogs found!');
        return;
    }

    // Set medical as default (or first catalog if medical doesn't exist)
    const medicalCatalog = catalogs.find(c => c.slug === 'medical') || catalogs[0];

    // Update all catalogs to not be default first
    await prisma.catalog.updateMany({
        where: {},
        data: { isDefault: false }
    });

    // Set the chosen catalog as default
    await prisma.catalog.update({
        where: { id: medicalCatalog.id },
        data: { isDefault: true }
    });

    console.log(`✅ Set ${medicalCatalog.name} as default catalog\n`);

    // Fix products with NULL catalog_id
    const nullProducts = await prisma.product.findMany({
        where: { catalogId: null },
        include: { category: true }
    });

    if (nullProducts.length > 0) {
        console.log(`Found ${nullProducts.length} products with NULL catalog_id:`);

        for (const product of nullProducts) {
            // Try to get catalog from category
            let catalogId = product.category?.catalogId;

            // If category also has no catalog, use default
            if (!catalogId) {
                catalogId = medicalCatalog.id;
            }

            await prisma.product.update({
                where: { id: product.id },
                data: { catalogId }
            });

            console.log(`  ✅ Fixed product "${product.name}" → ${catalogId === medicalCatalog.id ? 'default catalog' : 'category catalog'}`);
        }
    } else {
        console.log('✅ No products with NULL catalog_id');
    }

    // Fix categories with NULL catalog_id
    const nullCategories = await prisma.category.findMany({
        where: { catalogId: null }
    });

    if (nullCategories.length > 0) {
        await prisma.category.updateMany({
            where: { catalogId: null },
            data: { catalogId: medicalCatalog.id }
        });
        console.log(`\n✅ Fixed ${nullCategories.length} categories with NULL catalog_id`);
    } else {
        console.log('\n✅ No categories with NULL catalog_id');
    }

    console.log('\n=== COMPLETE ===');
    console.log('Run diagnose-catalog-isolation.ts to verify the fixes.');
}

setDefaultCatalogAndFixNulls()
    .catch(console.error)
    .finally(() => prisma.$disconnect());
