"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.inventoryService = exports.InventoryService = void 0;
const database_1 = __importDefault(require("../config/database"));
class InventoryService {
    // ==========================================
    // SUPPLIER INVENTORY
    // ==========================================
    /**
     * Get current stock level for a product
     */
    async checkSupplierStock(productId) {
        const inventory = await database_1.default.supplierInventory.findUnique({
            where: { productId }
        });
        return inventory?.quantity || 0;
    }
    /**
   * Add stock to supplier inventory (Purchase, Adjustment, Returns)
   */
    async addSupplierStock(productId, quantity, notes = 'Manual Adjustment', type = 'ADJUSTMENT' // Use string literal to avoid runtime Enum lookup issues
    ) {
        return await database_1.default.$transaction(async (tx) => {
            // 1. Update or Create Inventory Record
            const inventory = await tx.supplierInventory.upsert({
                where: { productId },
                create: {
                    productId,
                    quantity,
                    lastRestockDate: new Date()
                },
                update: {
                    quantity: { increment: quantity },
                    lastRestockDate: new Date()
                }
            });
            // 2. Create Audit Log
            await tx.stockTransaction.create({
                data: {
                    productId,
                    type,
                    quantity,
                    toLocationType: 'SUPPLIER',
                    notes
                }
            });
            // 3. Update Legacy Product Stock Field (Backward Compatibility)
            await tx.product.update({
                where: { id: productId },
                data: { stockQuantity: inventory.quantity }
            });
            return inventory;
        });
    }
    /**
     * Deduct stock from supplier inventory
     */
    async deductSupplierStock(productId, quantity, referenceId, notes = 'Deduction') {
        return await database_1.default.$transaction(async (tx) => {
            // 1. Check stock
            const current = await tx.supplierInventory.findUnique({ where: { productId } });
            if (!current || current.quantity < quantity) {
                throw new Error(`Insufficient stock for product ${productId}. Requested: ${quantity}, Available: ${current?.quantity || 0}`);
            }
            // 2. Deduct
            const inventory = await tx.supplierInventory.update({
                where: { productId },
                data: { quantity: { decrement: quantity } }
            });
            // 3. Audit Log
            await tx.stockTransaction.create({
                data: {
                    productId,
                    type: 'SALE',
                    quantity: -quantity,
                    fromLocationType: 'SUPPLIER',
                    referenceId, // Order ID usually
                    notes
                }
            });
            // 4. Update Legacy Product Stock Field
            await tx.product.update({
                where: { id: productId },
                data: { stockQuantity: inventory.quantity }
            });
            return inventory;
        });
    }
    /**
     * Get products below reorder level
     */
    async getLowStockProducts(threshold) {
        // If threshold provided, use it, else use product's own reorderLevel
        const whereClause = threshold !== undefined
            ? { quantity: { lte: threshold } }
            : { quantity: { lte: database_1.default.supplierInventory.fields.reorderLevel } };
        return await database_1.default.supplierInventory.findMany({
            where: whereClause,
            include: { product: true }
        });
    }
    // ==========================================
    // CUSTOMER INVENTORY
    // ==========================================
    /**
     * Get customer's inventory
     */
    async getCustomerInventory(customerId) {
        return await database_1.default.customerInventory.findMany({
            where: { customerId },
            include: { product: true, sourceOrder: true }
        });
    }
    /**
     * Add to customer inventory (usually from order delivery)
     */
    async addToCustomerInventory(customerId, productId, quantity, sourceOrderId) {
        return await database_1.default.$transaction(async (tx) => {
            // Create new entry for each receipt to track source order
            // We could aggregate by product, but requirements say "received date" and "source order" are important
            // So let's keep separate entries or aggregated?
            // "View their inventory dashboard... track received orders"
            // Suggestion: CustomerInventory table seems designed to hold specific receipt batches or aggregated?
            // Schema has sourceOrderId. If we want aggregate view, we query sum.
            // Let's create a new record for this specific receipt logic or update if same order/product combo exists?
            const inventory = await tx.customerInventory.create({
                data: {
                    customerId,
                    productId,
                    quantity,
                    sourceOrderId,
                    receivedAt: new Date()
                }
            });
            await tx.stockTransaction.create({
                data: {
                    productId,
                    type: 'TRANSFER',
                    quantity,
                    toLocationType: 'CUSTOMER',
                    toLocationId: customerId,
                    referenceId: sourceOrderId,
                    notes: 'Received from Order'
                }
            });
            return inventory;
        });
    }
    // ==========================================
    // STOCK TRANSFER (ORDER FULFILLMENT)
    // ==========================================
    /**
     * Validates if we can fulfill an order
     */
    async validateOrderStock(orderItems) {
        const outOfStockItems = [];
        for (const item of orderItems) {
            const stock = await this.checkSupplierStock(item.productId);
            if (stock < item.quantity) {
                outOfStockItems.push({
                    productId: item.productId,
                    productName: item.productName,
                    requested: item.quantity,
                    available: stock
                });
            }
        }
        return {
            isValid: outOfStockItems.length === 0,
            outOfStockItems
        };
    }
    /**
     * CORE: Process stock transfer for an Order
     * Deducts from Supplier -> Adds to Customer
     */
    async processOrderStockTransfer(orderId) {
        const order = await database_1.default.order.findUnique({
            where: { id: orderId },
            include: { items: true }
        });
        if (!order)
            throw new Error('Order not found');
        if (!order.userId)
            throw new Error('Order has no customer assigned');
        // 1. Validate first
        const validation = await this.validateOrderStock(order.items);
        if (!validation.isValid) {
            return {
                success: false,
                message: 'Insufficient stock',
                errors: validation.outOfStockItems.map(i => `${i.productName}: Need ${i.requested}, have ${i.available}`)
            };
        }
        try {
            await database_1.default.$transaction(async (tx) => {
                for (const item of order.items) {
                    if (item.inventoryProcessed)
                        continue; // Skip if already processed
                    // A. Deduct from Supplier
                    // Verify again inside transaction for safety
                    const supplierStock = await tx.supplierInventory.findUnique({ where: { productId: item.productId } });
                    if (!supplierStock || supplierStock.quantity < item.quantity) {
                        throw new Error(`Stock changed during processing for ${item.productName}`);
                    }
                    await tx.supplierInventory.update({
                        where: { productId: item.productId },
                        data: { quantity: { decrement: item.quantity } }
                    });
                    // Sync legacy field
                    await tx.product.update({
                        where: { id: item.productId },
                        data: { stockQuantity: { decrement: item.quantity } }
                    });
                    // B. Add to Customer
                    await tx.customerInventory.create({
                        data: {
                            customerId: order.userId,
                            productId: item.productId,
                            quantity: item.quantity,
                            sourceOrderId: order.id
                        }
                    });
                    // C. Log Transaction (Transfer)
                    await tx.stockTransaction.create({
                        data: {
                            productId: item.productId,
                            type: 'TRANSFER',
                            quantity: item.quantity,
                            fromLocationType: 'SUPPLIER',
                            toLocationType: 'CUSTOMER',
                            toLocationId: order.userId,
                            referenceId: order.id,
                            notes: `Order fulfillment ${order.orderNumber}`
                        }
                    });
                    // D. Mark item as processed
                    await tx.orderItem.update({
                        where: { id: item.id },
                        data: { inventoryProcessed: true, stockTransactionId: 'BATCH' } // Ideal: store specific transaction ID
                    });
                }
            });
            return { success: true, message: 'Stock transferred successfully' };
        }
        catch (error) {
            console.error('Stock transfer failed:', error);
            return { success: false, message: error.message };
        }
    }
    /**
     * Get transaction history
     */
    async getStockTransactionHistory(filters) {
        return await database_1.default.stockTransaction.findMany({
            where: filters, // Need to map specific filters strictly
            include: { product: true },
            orderBy: { createdAt: 'desc' }
        });
    }
}
exports.InventoryService = InventoryService;
exports.inventoryService = new InventoryService();
//# sourceMappingURL=inventory.service.js.map