"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.authorize = exports.authenticate = exports.PERMISSIONS = exports.ROLES = void 0;
const jsonwebtoken_1 = __importDefault(require("jsonwebtoken"));
const errorHandler_js_1 = require("./errorHandler.js");
exports.ROLES = {
    SUPER_ADMIN: 'SUPER_ADMIN',
    ADMIN: 'ADMIN', // Legacy, treated as Super Admin
    SALES_ADMIN: 'SALES_ADMIN',
    CATALOG_ADMIN: 'CATALOG_ADMIN',
    ACCOUNTING_ADMIN: 'ACCOUNTING_ADMIN',
    SUPPORT_ADMIN: 'SUPPORT_ADMIN',
    MANAGER: 'MANAGER',
    CUSTOMER: 'CUSTOMER'
};
exports.PERMISSIONS = {
    // Groups
    ALL_ADMINS: ['SUPER_ADMIN', 'ADMIN', 'SALES_ADMIN', 'CATALOG_ADMIN', 'ACCOUNTING_ADMIN', 'SUPPORT_ADMIN', 'MANAGER'],
    FULL_ACCESS: ['SUPER_ADMIN', 'ADMIN'],
    CATALOG_MANAGE: ['SUPER_ADMIN', 'ADMIN', 'CATALOG_ADMIN'],
    SALES_MANAGE: ['SUPER_ADMIN', 'ADMIN', 'SALES_ADMIN'],
    SALES_VIEW: ['SUPER_ADMIN', 'ADMIN', 'SALES_ADMIN', 'SUPPORT_ADMIN'], // Support can view
    ACCOUNTING_MANAGE: ['SUPER_ADMIN', 'ADMIN', 'ACCOUNTING_ADMIN'], // Re-check if Accounting is read-only? User said "Read-only access to accounting". But typically they might need to update status. I will assume Manage for now for "Accounting Admin" unless specified strictly read-only for EVERYTHING. Actually "Accounting Admin: Read-only access to accounting, invoices...". okay, I will make a ACCOUNTING_VIEW group.
    ACCOUNTING_VIEW: ['SUPER_ADMIN', 'ADMIN', 'ACCOUNTING_ADMIN', 'SALES_ADMIN'], // Sales can see invoices? "Sales Admin: ... invoices". Yes.
    CUSTOMER_MANAGE: ['SUPER_ADMIN', 'ADMIN', 'SALES_ADMIN'],
    CUSTOMER_VIEW: ['SUPER_ADMIN', 'ADMIN', 'SALES_ADMIN', 'SUPPORT_ADMIN'],
    SETTINGS_MANAGE: ['SUPER_ADMIN', 'ADMIN'],
};
const authenticate = async (req, res, next) => {
    try {
        const authHeader = req.headers.authorization;
        if (!authHeader || !authHeader.startsWith('Bearer ')) {
            throw (0, errorHandler_js_1.createError)('No token provided', 401);
        }
        const token = authHeader.substring(7);
        const secret = process.env.JWT_SECRET;
        if (!secret) {
            throw (0, errorHandler_js_1.createError)('JWT secret not configured', 500);
        }
        const decoded = jsonwebtoken_1.default.verify(token, secret);
        req.user = decoded;
        next();
    }
    catch (error) {
        if (error instanceof jsonwebtoken_1.default.JsonWebTokenError) {
            next((0, errorHandler_js_1.createError)('Invalid token', 401));
        }
        else if (error instanceof jsonwebtoken_1.default.TokenExpiredError) {
            next((0, errorHandler_js_1.createError)('Token expired', 401));
        }
        else {
            next(error);
        }
    }
};
exports.authenticate = authenticate;
const authorize = (...roles) => {
    return (req, res, next) => {
        if (!req.user) {
            return next((0, errorHandler_js_1.createError)('Not authenticated', 401));
        }
        if (!roles.includes(req.user.role)) {
            return next((0, errorHandler_js_1.createError)('Not authorized to access this resource', 403));
        }
        next();
    };
};
exports.authorize = authorize;
//# sourceMappingURL=auth.js.map