"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateSettings = exports.getSettings = void 0;
const database_js_1 = __importDefault(require("../config/database.js"));
/**
 * Get all system settings
 */
const getSettings = async (req, res, next) => {
    try {
        const settings = await database_js_1.default.systemSetting.findMany();
        // Default structure matching frontend state
        const formattedSettings = {
            notifications: {},
            business: {},
            system: {},
        };
        settings.forEach(setting => {
            // Parse value if it's JSON-like (booleans, numbers stored as strings)
            let parsedValue = setting.value;
            try {
                parsedValue = JSON.parse(setting.value);
            }
            catch (e) {
                // value is a simple string
            }
            if (!formattedSettings[setting.group]) {
                formattedSettings[setting.group] = {};
            }
            // key in DB is e.g. "business_email"
            // we want "email"
            const simpleKey = setting.key.replace(`${setting.group}_`, '');
            formattedSettings[setting.group][simpleKey] = parsedValue;
        });
        res.json({
            success: true,
            data: formattedSettings
        });
    }
    catch (error) {
        next(error);
    }
};
exports.getSettings = getSettings;
/**
 * Update system settings
 * Receives the full settings object or partial updates
 */
const updateSettings = async (req, res, next) => {
    try {
        const settingsData = req.body;
        const updates = [];
        // Iterate through groups (business, notifications, system)
        Object.keys(settingsData).forEach(group => {
            const groupSettings = settingsData[group];
            if (typeof groupSettings === 'object' && groupSettings !== null) {
                // Iterate through keys in the group
                Object.keys(groupSettings).forEach(key => {
                    const value = groupSettings[key];
                    const dbKey = `${group}_${key}`;
                    // JSON stringify the value for storage/consistency
                    const stringValue = JSON.stringify(value);
                    updates.push(database_js_1.default.systemSetting.upsert({
                        where: { key: dbKey },
                        update: {
                            value: stringValue,
                            group: group,
                        },
                        create: {
                            key: dbKey,
                            value: stringValue,
                            group: group,
                        },
                    }));
                });
            }
        });
        await Promise.all(updates);
        res.json({
            success: true,
            message: 'Settings updated successfully'
        });
    }
    catch (error) {
        next(error);
    }
};
exports.updateSettings = updateSettings;
//# sourceMappingURL=settings.controller.js.map