"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.uploadPlugin = exports.updatePluginConfig = exports.getPluginConfig = exports.getNavigation = exports.togglePlugin = exports.getPlugins = void 0;
// Note: In a real DI system we'd inject this. For now we might need to export the instance from server.ts
// OR we attach it to req.app.locals
// But typically for this codebase we can attach it to the Express app.
const getPlugins = async (req, res) => {
    try {
        console.log('[DEBUG] getPlugins controller called');
        // Access plugin manager from app locals (we will set this in server.ts)
        const pluginManager = req.app.get('pluginManager');
        console.log('[DEBUG] pluginManager retrieved:', !!pluginManager);
        if (!pluginManager) {
            console.error('[DEBUG] Plugin Manager missing');
            return res.status(500).json({ success: false, error: 'Plugin Manager not initialized' });
        }
        console.log('[DEBUG] Calling pluginManager.getPlugins()');
        // Check if getPlugins exists
        if (typeof pluginManager.getPlugins !== 'function') {
            console.error('[DEBUG] getPlugins is not a function on pluginManager:', pluginManager);
            throw new Error('pluginManager.getPlugins is not a function');
        }
        const rawPlugins = pluginManager.getPlugins();
        console.log('[DEBUG] rawPlugins:', rawPlugins ? rawPlugins.length : 'null');
        const plugins = rawPlugins.map((p) => ({
            name: p.name,
            version: p.version,
            status: p.status, // ACTIVE or INACTIVE
            description: p.description || 'System Plugin'
        }));
        res.json({
            success: true,
            data: plugins
        });
    }
    catch (error) {
        console.error('Get plugins error:', error);
        res.status(500).json({ success: false, error: 'Failed to fetch plugins' });
    }
};
exports.getPlugins = getPlugins;
const togglePlugin = async (req, res) => {
    try {
        const { name } = req.params;
        // const { enabled } = req.body; // Later we can pass explicit state
        const pluginManager = req.app.get('pluginManager');
        if (!pluginManager) {
            return res.status(500).json({ success: false, error: 'Plugin Manager not initialized' });
        }
        const plugin = pluginManager.getPlugins().find(p => p.name === name);
        if (!plugin) {
            return res.status(404).json({ success: false, error: 'Plugin not found' });
        }
        const newState = plugin.status !== 'ACTIVE'; // Toggle
        await pluginManager.togglePlugin(name, newState);
        res.json({
            success: true,
            data: {
                name,
                status: newState ? 'ACTIVE' : 'INACTIVE',
                message: 'Restart required to apply changes completely.'
            }
        });
    }
    catch (error) {
        console.error('Toggle plugin error:', error);
        res.status(500).json({ success: false, error: 'Failed to toggle plugin' });
    }
};
exports.togglePlugin = togglePlugin;
const getNavigation = async (req, res) => {
    try {
        const pluginManager = req.app.get('pluginManager');
        if (!pluginManager)
            return res.json({ success: true, data: [] });
        const nav = pluginManager.getNavigation();
        res.json({ success: true, data: nav });
    }
    catch (err) {
        res.status(500).json({ success: false, error: 'Failed to get nav' });
    }
};
exports.getNavigation = getNavigation;
const getPluginConfig = async (req, res) => {
    try {
        const { name } = req.params;
        const pluginManager = req.app.get('pluginManager');
        const config = await pluginManager.getPluginConfig(name);
        res.json({ success: true, data: config });
    }
    catch (err) {
        res.status(500).json({ success: false, error: 'Failed to get plugin config' });
    }
};
exports.getPluginConfig = getPluginConfig;
const updatePluginConfig = async (req, res) => {
    try {
        const { name } = req.params;
        const config = req.body;
        const pluginManager = req.app.get('pluginManager');
        await pluginManager.updatePluginConfig(name, config);
        res.json({ success: true, message: 'Configuration updated' });
    }
    catch (err) {
        res.status(500).json({ success: false, error: 'Failed to update plugin config' });
    }
};
exports.updatePluginConfig = updatePluginConfig;
const uploadPlugin = async (req, res) => {
    try {
        if (!req.file) {
            return res.status(400).json({ success: false, error: 'No file uploaded' });
        }
        const pluginManager = req.app.get('pluginManager');
        await pluginManager.installPlugin(req.file.path);
        res.json({
            success: true,
            message: 'Plugin uploaded and installed. Please restart the server to load it.'
        });
    }
    catch (err) {
        console.error('Upload Error:', err);
        res.status(500).json({ success: false, error: 'Failed to upload plugin' });
    }
};
exports.uploadPlugin = uploadPlugin;
//# sourceMappingURL=plugin.controller.js.map