"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.resetCustomerPassword = exports.deleteCustomer = exports.updateCustomer = exports.getCustomerById = exports.getAllCustomers = exports.createCustomer = void 0;
const bcryptjs_1 = __importDefault(require("bcryptjs"));
const database_js_1 = __importDefault(require("../config/database.js"));
const errorHandler_js_1 = require("../middleware/errorHandler.js");
const email_service_js_1 = require("../services/email.service.js");
const catalogContext_js_1 = require("../utils/catalogContext.js");
/**
 * Generate a random password
 */
const generatePassword = (length = 12) => {
    const charset = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789!@#$%^&*';
    let password = '';
    for (let i = 0; i < length; i++) {
        password += charset.charAt(Math.floor(Math.random() * charset.length));
    }
    return password;
};
/**
 * Create a new customer (Admin only)
 */
const createCustomer = async (req, res, next) => {
    try {
        const { email, username, password, firstName, lastName, phone, companyName, rneNumber, rnePdfUrl, taxId, customerType, address, 
        // B2B Financial Fields
        paymentMethod, paymentTerm, financialLimit, accountStatus, } = req.body;
        // Validation
        if (!email) {
            throw (0, errorHandler_js_1.createError)('Email is required', 400);
        }
        if (!firstName || !lastName) {
            throw (0, errorHandler_js_1.createError)('First name and last name are required', 400);
        }
        if (!companyName) {
            throw (0, errorHandler_js_1.createError)('Company name is required', 400);
        }
        if (!rneNumber) {
            throw (0, errorHandler_js_1.createError)('Commercial Registration Number (RNE) is required', 400);
        }
        // Check if user already exists
        const existingUser = await database_js_1.default.user.findFirst({
            where: {
                OR: [
                    { email },
                    username ? { username } : {},
                    { rneNumber },
                ],
            },
        });
        if (existingUser) {
            if (existingUser.email === email) {
                throw (0, errorHandler_js_1.createError)('A customer with this email already exists', 409);
            }
            if (existingUser.username === username) {
                throw (0, errorHandler_js_1.createError)('This username is already taken', 409);
            }
            if (existingUser.rneNumber === rneNumber) {
                throw (0, errorHandler_js_1.createError)('A customer with this RNE number already exists', 409);
            }
        }
        // Generate or use provided password
        const plainPassword = password || generatePassword();
        const passwordHash = await bcryptjs_1.default.hash(plainPassword, 10);
        // Resolve catalog context
        const catalog = await (0, catalogContext_js_1.resolveCatalogContext)(req, database_js_1.default);
        // Map paymentMethod to paymentTerm if needed
        let finalPaymentTerm = paymentTerm;
        if (!finalPaymentTerm && paymentMethod) {
            // Map old paymentMethod values to paymentTerm enum
            const paymentMethodMap = {
                'COD': 'IMMEDIATE',
                'CHEQUE': 'IMMEDIATE',
                'IMMEDIATE': 'IMMEDIATE',
            };
            finalPaymentTerm = paymentMethodMap[paymentMethod] || paymentMethod;
        }
        // Create customer
        const customer = await database_js_1.default.user.create({
            data: {
                email,
                username: username || email.split('@')[0],
                passwordHash,
                firstName,
                lastName,
                phone,
                companyName,
                rneNumber,
                rnePdfUrl,
                taxId,
                customerType,
                role: 'CUSTOMER',
                isActive: true,
                emailVerified: true, // Admin-created accounts are pre-verified
                // B2B Financial Fields
                paymentTerm: finalPaymentTerm || 'IMMEDIATE',
                financialLimit: financialLimit !== undefined ? Number(financialLimit) : 0,
                currentOutstanding: 0,
                accountStatus: accountStatus || 'COMMERCIAL_IN_PROCESS',
                catalogAssignments: {
                    create: {
                        catalogId: catalog.id,
                    },
                },
            },
            select: {
                id: true,
                email: true,
                username: true,
                firstName: true,
                lastName: true,
                phone: true,
                companyName: true,
                rneNumber: true,
                rnePdfUrl: true,
                taxId: true,
                customerType: true,
                role: true,
                isActive: true,
                // B2B Financial Fields
                paymentTerm: true,
                financialLimit: true,
                currentOutstanding: true,
                accountStatus: true,
                createdAt: true,
            },
        });
        // Create address if provided
        if (address) {
            await database_js_1.default.address.create({
                data: {
                    userId: customer.id,
                    label: 'Company Address',
                    street: address.street,
                    city: address.city,
                    state: address.state,
                    postalCode: address.postalCode,
                    country: address.country || 'Tunisia',
                    isDefault: true,
                },
            });
        }
        // Send credentials via email
        try {
            await (0, email_service_js_1.sendCustomerCredentials)({
                customerEmail: customer.email,
                customerName: `${customer.firstName} ${customer.lastName}`,
                username: customer.username,
                password: plainPassword,
                companyName: customer.companyName || undefined,
            });
        }
        catch (emailError) {
            console.error('Failed to send credentials email:', emailError);
            // Don't fail the request if email fails, but log it
        }
        res.status(201).json({
            success: true,
            message: 'Customer created successfully',
            data: {
                customer,
                credentials: {
                    username: customer.username,
                    password: plainPassword, // Return password to admin for display
                    email: customer.email,
                },
            },
        });
    }
    catch (error) {
        next(error);
    }
};
exports.createCustomer = createCustomer;
/**
 * Get all customers (Admin only)
 */
const getAllCustomers = async (req, res, next) => {
    try {
        const { page = 1, limit = 20, search, customerType, isActive } = req.query;
        const skip = (Number(page) - 1) * Number(limit);
        const take = Number(limit);
        // Build where clause
        const catalog = await (0, catalogContext_js_1.resolveCatalogContext)(req, database_js_1.default);
        const where = {
            role: 'CUSTOMER',
            catalogAssignments: {
                some: {
                    catalogId: catalog.id,
                },
            },
        };
        if (search) {
            where.OR = [
                { email: { contains: search, mode: 'insensitive' } },
                { firstName: { contains: search, mode: 'insensitive' } },
                { lastName: { contains: search, mode: 'insensitive' } },
                { companyName: { contains: search, mode: 'insensitive' } },
                { rneNumber: { contains: search, mode: 'insensitive' } },
            ];
        }
        if (customerType) {
            where.customerType = customerType;
        }
        if (isActive !== undefined) {
            where.isActive = isActive === 'true';
        }
        // Get customers with pagination
        const [customers, total] = await Promise.all([
            database_js_1.default.user.findMany({
                where,
                skip,
                take,
                select: {
                    id: true,
                    email: true,
                    username: true,
                    firstName: true,
                    lastName: true,
                    phone: true,
                    companyName: true,
                    rneNumber: true,
                    rnePdfUrl: true,
                    taxId: true,
                    customerType: true,
                    isActive: true,
                    emailVerified: true,
                    // B2B Financial Fields
                    paymentTerm: true,
                    financialLimit: true,
                    currentOutstanding: true,
                    accountStatus: true,
                    createdAt: true,
                    updatedAt: true,
                    _count: {
                        select: {
                            orders: true,
                        },
                    },
                },
                orderBy: {
                    createdAt: 'desc',
                },
            }),
            database_js_1.default.user.count({ where }),
        ]);
        res.json({
            success: true,
            data: {
                customers,
                pagination: {
                    page: Number(page),
                    limit: Number(limit),
                    total,
                    totalPages: Math.ceil(total / Number(limit)),
                },
            },
        });
    }
    catch (error) {
        next(error);
    }
};
exports.getAllCustomers = getAllCustomers;
/**
 * Get customer by ID (Admin only)
 */
const getCustomerById = async (req, res, next) => {
    try {
        const { id } = req.params;
        const customer = await database_js_1.default.user.findUnique({
            where: { id },
            select: {
                id: true,
                email: true,
                username: true,
                firstName: true,
                lastName: true,
                phone: true,
                companyName: true,
                rneNumber: true,
                rnePdfUrl: true,
                taxId: true,
                customerType: true,
                role: true,
                isActive: true,
                emailVerified: true,
                // B2B Financial Fields
                paymentTerm: true,
                financialLimit: true,
                currentOutstanding: true,
                accountStatus: true,
                createdAt: true,
                updatedAt: true,
                addresses: {
                    select: {
                        id: true,
                        label: true,
                        street: true,
                        city: true,
                        state: true,
                        postalCode: true,
                        country: true,
                        isDefault: true,
                    },
                },
                orders: {
                    select: {
                        id: true,
                        orderNumber: true,
                        status: true,
                        totalAmount: true,
                        createdAt: true,
                    },
                    orderBy: {
                        createdAt: 'desc',
                    },
                    take: 10,
                },
                _count: {
                    select: {
                        orders: true,
                    },
                },
            },
        });
        if (!customer) {
            throw (0, errorHandler_js_1.createError)('Customer not found', 404);
        }
        if (customer.role !== 'CUSTOMER') {
            throw (0, errorHandler_js_1.createError)('User is not a customer', 400);
        }
        res.json({
            success: true,
            data: customer,
        });
    }
    catch (error) {
        next(error);
    }
};
exports.getCustomerById = getCustomerById;
/**
 * Update customer (Admin only)
 */
const updateCustomer = async (req, res, next) => {
    try {
        const { id } = req.params;
        const { firstName, lastName, phone, companyName, rneNumber, rnePdfUrl, taxId, customerType, isActive, 
        // B2B Financial Fields
        paymentMethod, paymentTerm, financialLimit, currentOutstanding, accountStatus, } = req.body;
        // Check if customer exists
        const existingCustomer = await database_js_1.default.user.findUnique({
            where: { id },
        });
        if (!existingCustomer) {
            throw (0, errorHandler_js_1.createError)('Customer not found', 404);
        }
        if (existingCustomer.role !== 'CUSTOMER') {
            throw (0, errorHandler_js_1.createError)('User is not a customer', 400);
        }
        // Check for duplicate RNE if updating
        if (rneNumber && rneNumber !== existingCustomer.rneNumber) {
            const duplicateRNE = await database_js_1.default.user.findFirst({
                where: {
                    rneNumber,
                    id: { not: id },
                },
            });
            if (duplicateRNE) {
                throw (0, errorHandler_js_1.createError)('A customer with this RNE number already exists', 409);
            }
        }
        // Map paymentMethod to paymentTerm if needed
        let finalPaymentTerm = paymentTerm;
        if (!finalPaymentTerm && paymentMethod) {
            const paymentMethodMap = {
                'COD': 'IMMEDIATE',
                'CHEQUE': 'IMMEDIATE',
                'IMMEDIATE': 'IMMEDIATE',
            };
            finalPaymentTerm = paymentMethodMap[paymentMethod] || paymentMethod;
        }
        // Prepare update data
        const updateData = {
            firstName,
            lastName,
            phone,
            companyName,
            rneNumber,
            rnePdfUrl,
            taxId,
            customerType,
            isActive,
        };
        // Add financial fields if provided
        if (finalPaymentTerm !== undefined)
            updateData.paymentTerm = finalPaymentTerm;
        if (financialLimit !== undefined)
            updateData.financialLimit = Number(financialLimit);
        if (currentOutstanding !== undefined)
            updateData.currentOutstanding = Number(currentOutstanding);
        if (accountStatus !== undefined)
            updateData.accountStatus = accountStatus;
        // Update customer
        const updatedCustomer = await database_js_1.default.user.update({
            where: { id },
            data: updateData,
            select: {
                id: true,
                email: true,
                username: true,
                firstName: true,
                lastName: true,
                phone: true,
                companyName: true,
                rneNumber: true,
                rnePdfUrl: true,
                taxId: true,
                customerType: true,
                isActive: true,
                // B2B Financial Fields
                paymentTerm: true,
                financialLimit: true,
                currentOutstanding: true,
                accountStatus: true,
                updatedAt: true,
            },
        });
        res.json({
            success: true,
            message: 'Customer updated successfully',
            data: updatedCustomer,
        });
    }
    catch (error) {
        next(error);
    }
};
exports.updateCustomer = updateCustomer;
/**
 * Delete customer (Admin only)
 */
const deleteCustomer = async (req, res, next) => {
    try {
        const { id } = req.params;
        // Check if customer exists
        const customer = await database_js_1.default.user.findUnique({
            where: { id },
            include: {
                _count: {
                    select: {
                        orders: true,
                    },
                },
            },
        });
        if (!customer) {
            throw (0, errorHandler_js_1.createError)('Customer not found', 404);
        }
        if (customer.role !== 'CUSTOMER') {
            throw (0, errorHandler_js_1.createError)('User is not a customer', 400);
        }
        // Check if customer has orders
        if (customer._count.orders > 0) {
            // Soft delete by deactivating instead of hard delete
            await database_js_1.default.user.update({
                where: { id },
                data: { isActive: false },
            });
            return res.json({
                success: true,
                message: 'Customer has orders and has been deactivated instead of deleted',
            });
        }
        // Hard delete if no orders
        await database_js_1.default.user.delete({
            where: { id },
        });
        res.json({
            success: true,
            message: 'Customer deleted successfully',
        });
    }
    catch (error) {
        next(error);
    }
};
exports.deleteCustomer = deleteCustomer;
/**
 * Reset customer password (Admin only)
 */
const resetCustomerPassword = async (req, res, next) => {
    try {
        const { id } = req.params;
        const { password } = req.body;
        // Check if customer exists
        const customer = await database_js_1.default.user.findUnique({
            where: { id },
        });
        if (!customer) {
            throw (0, errorHandler_js_1.createError)('Customer not found', 404);
        }
        if (customer.role !== 'CUSTOMER') {
            throw (0, errorHandler_js_1.createError)('User is not a customer', 400);
        }
        // Generate or use provided password
        const plainPassword = password || generatePassword();
        const passwordHash = await bcryptjs_1.default.hash(plainPassword, 10);
        // Update password
        await database_js_1.default.user.update({
            where: { id },
            data: { passwordHash },
        });
        // Send new credentials via email
        try {
            await (0, email_service_js_1.sendCustomerCredentials)({
                customerEmail: customer.email,
                customerName: `${customer.firstName} ${customer.lastName}`,
                username: customer.username || customer.email,
                password: plainPassword,
                companyName: customer.companyName || undefined,
            });
        }
        catch (emailError) {
            console.error('Failed to send password reset email:', emailError);
        }
        res.json({
            success: true,
            message: 'Password reset successfully',
            data: {
                newPassword: plainPassword,
            },
        });
    }
    catch (error) {
        next(error);
    }
};
exports.resetCustomerPassword = resetCustomerPassword;
//# sourceMappingURL=customer.controller.js.map