"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.resetPassword = exports.forgotPassword = exports.getCurrentUser = exports.logout = exports.refreshToken = exports.login = exports.register = void 0;
const bcryptjs_1 = __importDefault(require("bcryptjs"));
const jsonwebtoken_1 = __importDefault(require("jsonwebtoken"));
const database_js_1 = __importDefault(require("../config/database.js"));
const errorHandler_js_1 = require("../middleware/errorHandler.js");
// Generate JWT token
const generateToken = (userId, email, role) => {
    const secret = process.env.JWT_SECRET;
    const expiresIn = process.env.JWT_EXPIRES_IN || '7d';
    if (!secret) {
        throw (0, errorHandler_js_1.createError)('JWT secret not configured', 500);
    }
    return jsonwebtoken_1.default.sign({ id: userId, email, role }, secret, { expiresIn });
};
// Generate refresh token
const generateRefreshToken = (userId) => {
    const secret = process.env.JWT_REFRESH_SECRET;
    const expiresIn = process.env.JWT_REFRESH_EXPIRES_IN || '30d';
    if (!secret) {
        throw (0, errorHandler_js_1.createError)('JWT refresh secret not configured', 500);
    }
    return jsonwebtoken_1.default.sign({ id: userId }, secret, { expiresIn });
};
// Register new user
const register = async (req, res, next) => {
    try {
        const { email, password, firstName, lastName, phone } = req.body;
        // Validation
        if (!email || !password) {
            throw (0, errorHandler_js_1.createError)('Email and password are required', 400);
        }
        if (password.length < 6) {
            throw (0, errorHandler_js_1.createError)('Password must be at least 6 characters', 400);
        }
        // Check if user exists
        const existingUser = await database_js_1.default.user.findUnique({
            where: { email },
        });
        if (existingUser) {
            throw (0, errorHandler_js_1.createError)('User already exists with this email', 409);
        }
        // Hash password
        const passwordHash = await bcryptjs_1.default.hash(password, 10);
        // Create user
        const user = await database_js_1.default.user.create({
            data: {
                email,
                passwordHash,
                firstName,
                lastName,
                phone,
                role: 'CUSTOMER',
            },
            select: {
                id: true,
                email: true,
                firstName: true,
                lastName: true,
                phone: true,
                role: true,
                paymentTerm: true,
                financialLimit: true,
                currentOutstanding: true,
                accountStatus: true,
                createdAt: true,
            },
        });
        // Generate tokens
        const token = generateToken(user.id, user.email, user.role);
        const refreshToken = generateRefreshToken(user.id);
        // Save refresh token
        await database_js_1.default.refreshToken.create({
            data: {
                token: refreshToken,
                userId: user.id,
                expiresAt: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000), // 30 days
            },
        });
        res.status(201).json({
            success: true,
            data: {
                user,
                token,
                refreshToken,
            },
        });
    }
    catch (error) {
        next(error);
    }
};
exports.register = register;
// Login user
const login = async (req, res, next) => {
    try {
        const { email, username, password } = req.body;
        // Validation
        if ((!email && !username) || !password) {
            throw (0, errorHandler_js_1.createError)('Email/Username and password are required', 400);
        }
        // Find user by email or username - INCLUDE CATALOG ASSIGNMENTS
        const user = await database_js_1.default.user.findFirst({
            where: {
                OR: [
                    email ? { email } : {},
                    username ? { username } : {},
                ],
            },
            include: {
                catalogAssignments: {
                    include: {
                        catalog: {
                            select: {
                                id: true,
                                name: true,
                                slug: true,
                                isActive: true,
                            }
                        }
                    }
                }
            }
        });
        if (!user) {
            throw (0, errorHandler_js_1.createError)('Invalid credentials', 401);
        }
        // Check if user is active
        if (!user.isActive) {
            throw (0, errorHandler_js_1.createError)('Account is deactivated', 403);
        }
        // Verify password
        const isPasswordValid = await bcryptjs_1.default.compare(password, user.passwordHash);
        if (!isPasswordValid) {
            throw (0, errorHandler_js_1.createError)('Invalid credentials', 401);
        }
        // Generate tokens
        const token = generateToken(user.id, user.email, user.role);
        const refreshToken = generateRefreshToken(user.id);
        // Save refresh token
        await database_js_1.default.refreshToken.create({
            data: {
                token: refreshToken,
                userId: user.id,
                expiresAt: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000),
            },
        });
        // Extract catalogId from assignments
        const catalogId = user.catalogAssignments?.[0]?.catalogId || null;
        res.json({
            success: true,
            data: {
                user: {
                    id: user.id,
                    email: user.email,
                    username: user.username,
                    firstName: user.firstName,
                    lastName: user.lastName,
                    phone: user.phone,
                    companyName: user.companyName,
                    customerType: user.customerType,
                    role: user.role,
                    paymentTerm: user.paymentTerm,
                    financialLimit: user.financialLimit,
                    currentOutstanding: user.currentOutstanding,
                    accountStatus: user.accountStatus,
                    language: user.language, // ✅ Add language
                    catalogId, // ✅ Add catalogId to response
                    catalogAssignments: user.catalogAssignments, // ✅ Add full assignments
                },
                token,
                refreshToken,
            },
        });
    }
    catch (error) {
        next(error);
    }
};
exports.login = login;
// Refresh token
const refreshToken = async (req, res, next) => {
    try {
        const { refreshToken } = req.body;
        if (!refreshToken) {
            throw (0, errorHandler_js_1.createError)('Refresh token is required', 400);
        }
        // Verify refresh token
        const secret = process.env.JWT_REFRESH_SECRET;
        if (!secret) {
            throw (0, errorHandler_js_1.createError)('JWT refresh secret not configured', 500);
        }
        const decoded = jsonwebtoken_1.default.verify(refreshToken, secret);
        // Check if refresh token exists in database
        const storedToken = await database_js_1.default.refreshToken.findUnique({
            where: { token: refreshToken },
            include: { user: true },
        });
        if (!storedToken || storedToken.expiresAt < new Date()) {
            throw (0, errorHandler_js_1.createError)('Invalid or expired refresh token', 401);
        }
        // Generate new tokens
        const newToken = generateToken(storedToken.user.id, storedToken.user.email, storedToken.user.role);
        const newRefreshToken = generateRefreshToken(storedToken.user.id);
        // Delete old refresh token and save new one
        await database_js_1.default.refreshToken.delete({
            where: { id: storedToken.id },
        });
        await database_js_1.default.refreshToken.create({
            data: {
                token: newRefreshToken,
                userId: storedToken.user.id,
                expiresAt: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000),
            },
        });
        res.json({
            success: true,
            data: {
                token: newToken,
                refreshToken: newRefreshToken,
            },
        });
    }
    catch (error) {
        next(error);
    }
};
exports.refreshToken = refreshToken;
// Logout
const logout = async (req, res, next) => {
    try {
        const { refreshToken } = req.body;
        if (refreshToken) {
            await database_js_1.default.refreshToken.deleteMany({
                where: { token: refreshToken },
            });
        }
        res.json({
            success: true,
            message: 'Logged out successfully',
        });
    }
    catch (error) {
        next(error);
    }
};
exports.logout = logout;
// Get current user
const getCurrentUser = async (req, res, next) => {
    try {
        if (!req.user) {
            throw (0, errorHandler_js_1.createError)('Not authenticated', 401);
        }
        const user = await database_js_1.default.user.findUnique({
            where: { id: req.user.id },
            select: {
                id: true,
                email: true,
                username: true,
                firstName: true,
                lastName: true,
                phone: true,
                companyName: true,
                rneNumber: true,
                taxId: true,
                customerType: true,
                role: true,
                paymentTerm: true,
                financialLimit: true,
                currentOutstanding: true,
                accountStatus: true,
                isActive: true,
                language: true, // ✅ Add language
                emailVerified: true,
                createdAt: true,
                catalogAssignments: {
                    include: {
                        catalog: {
                            select: {
                                id: true,
                                name: true,
                                slug: true,
                                isActive: true,
                            }
                        }
                    }
                }
            },
        });
        if (!user) {
            throw (0, errorHandler_js_1.createError)('User not found', 404);
        }
        // Extract catalogId from assignments
        const catalogId = user.catalogAssignments?.[0]?.catalogId || null;
        res.json({
            success: true,
            data: {
                ...user,
                catalogId, // ✅ Add catalogId
            },
        });
    }
    catch (error) {
        next(error);
    }
};
exports.getCurrentUser = getCurrentUser;
// Forgot password (placeholder - implement email sending)
const forgotPassword = async (req, res, next) => {
    try {
        const { email } = req.body;
        if (!email) {
            throw (0, errorHandler_js_1.createError)('Email is required', 400);
        }
        const user = await database_js_1.default.user.findUnique({
            where: { email },
        });
        if (!user) {
            // Don't reveal if user exists
            res.json({
                success: true,
                message: 'If the email exists, a password reset link has been sent',
            });
            return;
        }
        // TODO: Generate reset token and send email
        // For now, just return success
        res.json({
            success: true,
            message: 'If the email exists, a password reset link has been sent',
        });
    }
    catch (error) {
        next(error);
    }
};
exports.forgotPassword = forgotPassword;
// Reset password (placeholder)
const resetPassword = async (req, res, next) => {
    try {
        const { token, newPassword } = req.body;
        if (!token || !newPassword) {
            throw (0, errorHandler_js_1.createError)('Token and new password are required', 400);
        }
        if (newPassword.length < 6) {
            throw (0, errorHandler_js_1.createError)('Password must be at least 6 characters', 400);
        }
        // TODO: Verify reset token and update password
        res.json({
            success: true,
            message: 'Password reset successfully',
        });
    }
    catch (error) {
        next(error);
    }
};
exports.resetPassword = resetPassword;
//# sourceMappingURL=auth.controller.js.map