"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteAdmin = exports.updateAdminRole = exports.createAdmin = exports.listAdmins = void 0;
const bcryptjs_1 = __importDefault(require("bcryptjs"));
const database_js_1 = __importDefault(require("../config/database.js"));
const errorHandler_js_1 = require("../middleware/errorHandler.js");
const client_1 = require("@prisma/client");
// List all admins
const listAdmins = async (req, res, next) => {
    try {
        const admins = await database_js_1.default.user.findMany({
            where: {
                role: {
                    in: [
                        'ADMIN',
                        'SUPER_ADMIN',
                        'SALES_ADMIN',
                        'CATALOG_ADMIN',
                        'ACCOUNTING_ADMIN',
                        'SUPPORT_ADMIN',
                        'MANAGER'
                    ]
                }
            },
            select: {
                id: true,
                email: true,
                username: true,
                firstName: true,
                lastName: true,
                role: true,
                isActive: true,
                createdAt: true
            },
            orderBy: { createdAt: 'desc' }
        });
        res.json({
            success: true,
            data: admins
        });
    }
    catch (error) {
        next(error);
    }
};
exports.listAdmins = listAdmins;
// Create new admin
const createAdmin = async (req, res, next) => {
    try {
        const { email, password, firstName, lastName, role, phone } = req.body;
        if (!email || !password || !role) {
            throw (0, errorHandler_js_1.createError)('Email, password, and role are required', 400);
        }
        if (!Object.values(client_1.UserRole).includes(role)) {
            throw (0, errorHandler_js_1.createError)('Invalid role', 400);
        }
        const existingUser = await database_js_1.default.user.findUnique({
            where: { email }
        });
        if (existingUser) {
            throw (0, errorHandler_js_1.createError)('User already exists', 409);
        }
        const passwordHash = await bcryptjs_1.default.hash(password, 10);
        const admin = await database_js_1.default.user.create({
            data: {
                email,
                passwordHash,
                firstName,
                lastName,
                phone,
                role: role,
                isActive: true
            },
            select: {
                id: true,
                email: true,
                role: true,
                isActive: true
            }
        });
        res.status(201).json({
            success: true,
            data: admin
        });
    }
    catch (error) {
        next(error);
    }
};
exports.createAdmin = createAdmin;
// Update admin role
const updateAdminRole = async (req, res, next) => {
    try {
        const { id } = req.params;
        const { role } = req.body;
        if (!role || !Object.values(client_1.UserRole).includes(role)) {
            throw (0, errorHandler_js_1.createError)('Valid role is required', 400);
        }
        // Prevent changing own role to avoid lockout (optional safety)
        if (req.user?.id === id) {
            // Allow if super admin? Maybe restrict self-demotion
        }
        const admin = await database_js_1.default.user.update({
            where: { id },
            data: { role: role },
            select: {
                id: true,
                email: true,
                role: true
            }
        });
        res.json({
            success: true,
            data: admin
        });
    }
    catch (error) {
        next(error);
    }
};
exports.updateAdminRole = updateAdminRole;
// Delete/Deactivate admin
const deleteAdmin = async (req, res, next) => {
    try {
        const { id } = req.params;
        if (req.user?.id === id) {
            throw (0, errorHandler_js_1.createError)('Cannot delete yourself', 400);
        }
        await database_js_1.default.user.delete({
            where: { id }
        });
        res.json({
            success: true,
            message: 'Admin deleted successfully'
        });
    }
    catch (error) {
        next(error);
    }
};
exports.deleteAdmin = deleteAdmin;
//# sourceMappingURL=admin.controller.js.map